[CmdletBinding()]
Param (
  [Parameter(Mandatory=$true)]
  [string]$appDefinition,
  [switch]$ignoreHashError
)

Set-StrictMode -Version latest
$ErrorActionPreference = 'Stop';
Get-Module | Where-Object {$_.ModuleType -eq 'Script'} | Remove-Module
import-module .\automateAppDeployment.psm1 

function Get-GlobaleDefinition {
  $globalDef=New-Module -AsCustomObject -ScriptBlock {
    [string]$site=$null
    [string]$siteServer=$null
    [string]$appDeploySource=$null
    [string]$targetBasePath=$null
    [string]$targetUNCBasePath=$null
    [string]$distributionPointGroupName=$null
    [string]$includeCollection=$null
    [string]$rootFolder=$null
    [int]$updateCollectionMin=0
    [string]$limitingCollection=$null
    [string]$namespace=$null
    [string]$fsTemplate=$null
    [string]$collectionPathTemplate=$null
  #  [string]$softwarePathTemplate=$null
    [string]$collectionNameTemplate=$null
    [string]$applicationNameTemplate=$null
    [string]$deploymentTypeTemplate=$null

    [bool]$doConfigMgrIntegration=$true
    Export-ModuleMember -Variable *
  }
  $globalDef.appDeploySource='D:\_Scripting\PSAppDeployToolkit\Toolkit'
  $globalDef.targetBasePath='D:\ConfigMgr\Content\Software' 
  $globalDef.targetUNCBasePath='\\cm01\ConfigMgr\Content\Software'
  $globalDef.rootFolder='_Software'
  $globalDef.DistributionPointGroupName='Intranet'
  $globalDef.includeCollection='Test Clients' 
  $globalDef.site='MB4'
  $globalDef.siteServer='cm01'
  $globalDef.updateCollectionMin=30
  $globalDef.limitingCollection='All Systems'
  $globalDef.namespace='root\SMS\Site_'+$globalDef.site
  $globalDef.doConfigMgrIntegration=$true

  $globalDef.fsTemplate='{Manufacturer}\{ProductName}\{ProductVersion}'
  $globalDef.collectionPathTemplate='{Manufacturer}'
  $globalDef.collectionNameTemplate='{ProductName} {ProductVersion} {bitrate} {languageID}'
  $globalDef.applicationNameTemplate='{ProductName} {ProductVersion} {bitrate} {languageID}' 
  $globalDef.deploymentTypeTemplate='{ProductName} {ProductVersion}' 
  $globalDef
}

<# function getAppName{
  [CmdletBinding()]
  param(
    [Parameter(Mandatory=$false)]
    [string]$vendor,
    [Parameter(Mandatory=$true)]
    [string]$software,
    [Parameter(Mandatory=$true)]
    [string]$version,
    [Parameter(Mandatory=$false)]
    [bool]$isX86
    
  )
  
  #sonderfall: wenn der Softwarename bereits die versionnummer beinhaltet, dann nicht nochmal hinzufuegen
  if ($software.Contains($version)) {
    $colName=$vendor + ' '+$software
  }
  else
  {
    $colName=$vendor+' '+$software + ' ' + $version
  }
  if ($isX86 -ne $null) {
    if ($isX86) {
      $colname+=' x86'
    }
    else
    {
      $colname+=' x64'
    }
  }
  $colName.trim() 
}#>

function Get-AppDefinition{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$false)]
    [xml]$xml,
    [Parameter(Mandatory=$false)]
    [string]$sourcefiles,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  $appDef = New-Module -AsCustomObject -ScriptBlock {
    [string]$Manufacturer=$null
    [string]$ProductName=$null
    [string]$ProductVersion=$null
    [string]$ProductCode=$null
    [bool]$isX86=$false
    [string]$bitrate=$null
    #path to source files (same as appdefinition.xml folder)
    [string]$sourcefiles=$null
    [string]$targetFiles=$null
    [string]$uncTargetFiles=$null
    #full path to main file
    [string]$mainFile=$null
    #only filename of main file
    [string]$mainFileShort=$null
    [string]$install=$null
    [string]$uninstall=$null
    [string]$detect=$null
    [string]$applicationObjectID=$null
    [string]$languageID=$null

    #name of hash algorithmen - currently only sha256
    [string]$hashType=$null
    #result of hash of main file to compare
    [string]$hashValue=$null

    #Format: Key: SHA256:234jk Value: uninstall=$true/$false
    [hashtable]$supersedence=@{}
    #Format: Key: Group Name, Value: [hashtable] Key: SHA256:234234, Value: autoinstall=$true/$False
    [hashtable]$dependency=@{}

    [bool]$hasUninstall=$true
    Export-ModuleMember -Variable *
  }
  
    #transfer values from xml file to object
  $appDef.sourcefiles=$sourcefiles
  
  $appDef.isX86=($xml.appdefinition.info.isX86 -ieq 'true')
  if ($appDef.isX86) {$appDef.bitrate='X86'} else {$appDef.bitrate='X64'}

  $appDef.hasUninstall=-not ($xml.appdefinition.info.hasUninstall -ieq 'false')
  
  $appDef.mainFile=Join-Path $appDef.sourceFiles ($xml.appdefinition.file)
  $appDef.mainFileShort=$xml.appdefinition.file
  if ($xml.appdefinition.info.setupType -eq 'MSI') {
    #extract msi table for infos...
    $msiTable=Get-MsiTableProperty -Path $appDef.mainFile -Table 'Property' -ContinueOnError $false
    
    $appDef.Manufacturer=$msiTable.Manufacturer 
    $appDef.ProductName=$msiTable.ProductName
    $appDef.ProductVersion=$msiTable.ProductVersion
    $appDef.ProductCode=$msiTable.ProductCode
    #region msp auswerten
  } elseif ($xml.appdefinition.info.setupType -eq 'MSP') {
    #source: http://www.software-virtualisierung.de/entry/msppowershell.html
        $wi = New-Object -com WindowsInstaller.Installer
        $mspdb = $wi.GetType().InvokeMember("OpenDatabase", "InvokeMethod", $Null, $wi, $($appDef.mainFile, 32))
        $su = $mspdb.GetType().InvokeMember("SummaryInformation", "GetProperty", $Null, $mspdb, $Null)
        #proerty 7: https://msdn.microsoft.com/en-us/library/windows/desktop/aa372043%28v=vs.85%29.aspx
        $appDef.ProductCode=$su.GetType().InvokeMember("Property", "GetProperty", $Null, $su, 7)
        #patchcode:
        #$su.GetType().InvokeMember("Property", "GetProperty", $Null, $su, 9)
        #{AC76BA86-7AD7-FFFF-2550-AE0F0675A700}{AC76BA86-7AD7-FFFF-2550-AE0F06756C00}{AC76BA86-7AD7-FFFF-2550-AE0F06758E00}{AC76BA86-7AD7-FFFF-2550-AE0F06759000}{AC76BA86-7AD7-FFFF-2550-AE0F06759100}
  }
  #endregion
  else { #exe auslesen
    $vers=(get-item $appDef.mainFile).VersionInfo
    if ($vers -ne $null) {
      if ($vers.CompanyName -ne $null) {$appDef.Manufacturer=$vers.CompanyName.trim()}
      if ($vers.ProductName -ne $null) {$appDef.ProductName=$vers.ProductName.trim()}
      if ($vers.ProductVersion -ne $null) {$appDef.ProductVersion=$vers.ProductVersion.trim()}
      if ($appDef.ProductVersion -eq $null) {
        $appDef.ProductVersion=$vers.FileVersion.trim()
      }
    }
  }
  


  #Company aus msi berschreiben, wenn es explizit im xml definiert ist
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'company' -and -not $_.IsEmpty} | ForEach-Object {$appDef.Manufacturer=$_.InnerText}
  #software aus msi berschreiben, wenn es explizit im xml definiert ist
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'productName' -and -not $_.IsEmpty} | ForEach-Object {$appDef.ProductName=$_.InnerText}
  #version aus msi berschreiben, wenn es explizit im xml definiert ist
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'productVersion' -and -not $_.IsEmpty} | ForEach-Object {$appDef.ProductVersion=$_.InnerText}
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'msiProductCode' -and -not $_.IsEmpty} | ForEach-Object {$appDef.ProductCode=$_.InnerText}
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'languageID' -and -not $_.IsEmpty} | ForEach-Object {$appDef.languageID=$_.InnerText}
  
  
  $xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'install' -and -not $_.IsEmpty} | ForEach-Object { $appDef.install=[Environment]::NewLine+($_.InnerText)}
  $xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'uninstall' -and -not $_.IsEmpty} | ForEach-Object { $appDef.uninstall=[Environment]::NewLine+($_.InnerText)}
  $xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'detection' -and -not $_.IsEmpty} | ForEach-Object { $appDef.detect=[Environment]::NewLine+($_.InnerText)}
  
  #read hash values
  $xml.appdefinition.ChildNodes | where {$_.name -eq "hash" -and -not $_.isEmpty} | ForEach-Object {
    $appdef.hashValue=$_.InnerText
    $appdef.hashType=$_.Type
  }

  $fsPart=replace-PathVariables -variable $globalDefObject.fsTemplate -appDefObject $appDef
  $appDef.uncTargetFiles=[IO.Path]::getFullPath((Join-Path $globalDefObject.targetUNCBasePath $fsPart))
  $appDef.targetFiles=[IO.Path]::getFullPath((Join-Path $globalDefObject.targetBasePath $fsPart))

  $supersedence=$xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'supersedence' -and -not $_.IsEmpty} 

  if ($supersedence -ne $null) {
    $xml.appdefinition.supersedence.deploymentType | ForEach-Object {
        $appdef.supersedence.add(($_.type+":"+$_.InnerText),($_.uninstall -ieq 'true'))
    }
  }

  $dependency=$xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'dependency' -and -not $_.IsEmpty} 

  write-host "dep: $dependency"
  if ($dependency -ne $null) {
   $dependency.ChildNodes  | Where-Object {$_.name -eq 'dependencyGroup' -and -not $_.IsEmpty} | ForEach-Object {
    $name=$_.name
    if ($appDef.dependency[$name] -eq $null) {
        [hashtable]$values=@{}
        $appDef.dependency.add($name,$values)
    }
    else
    {
        [hashtable]$values=$appDef.dependency.get[$name]
    }
    $_.application | ForEach-Object {
        $values.add(($_.type+":"+$_.InnerText),($_.autoInstall -ieq 'true'))
    }
  }
  }

  $appDef
}



function New-MSPSetup{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  ) 
  
  

  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    if (-not $appDefObject.isX86) {
      $appDefObject.detect='Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {$_.psChildname -like "'+$appDefObject.msiProductCode+'" -and $_.DisplayVersion -eq "'+$appDefObject.ProductVersion+'" -and $_.WindowsInstaller -eq 1}'
    }
    else
    {
      $appDefObject.detect='if ([IntPtr]::Size -eq 8) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
      $appDefObject.detect+=[Environment]::NewLine+' Get-ItemProperty $regPath | where {$_.psChildname -like "'+$appDefObject.msiProductCode+'" -and $_.DisplayVersion -eq "'+$appDefObject.ProductVersion+'" -and $_.WindowsInstaller -eq 1}'
    }
  }

  create-CMApplication -appDefObject $appDefObject -globalDefObject $globalDefObject

}


function New-MSISetup{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  ) 
  
  

  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    if (-not $appDefObject.isX86) {
      $appDefObject.detect='Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {$_.psChildname -like "'+$appDefObject.msiProductCode+'" -and $_.DisplayVersion -eq "'+$appDefObject.ProductVersion+'" -and $_.WindowsInstaller -eq 1}'
    }
    else
    {
      $appDefObject.detect='if ([IntPtr]::Size -eq 8) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
      $appDefObject.detect+=[Environment]::NewLine+' Get-ItemProperty $regPath | where {$_.psChildname -like "'+$appDefObject.msiProductCode+'" -and $_.DisplayVersion -eq "'+$appDefObject.ProductVersion+'" -and $_.WindowsInstaller -eq 1}'
    }
  }

  create-CMApplication -appDefObject $appDefObject -globalDefObject $globalDefObject

}

function New-InnoSetup {
  param
  (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  
  $mainFile=$appDefObject.mainFile
  

  if ($appDefObject.install -eq $null -or $appDefObject.install -eq '')
  {
    write-host 'No InnoSetup install parameter found. Generating script...'
    $appDefObject.install=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Installing '+($appDefObject.ProductName)+' '+($appDefObject.ProductVersion)+'. This may take some time. Please wait..."'
    $appDefObject.install+=[Environment]::NewLine+'$nsisLog=join-path $configToolkitLogDir ($installName + "_" + $appDeployToolkitName + "_" + $deploymentType + "_NSIS.log")'
    $appDefObject.install+=[Environment]::NewLine+'Execute-Process -Path "'+$appDefObject.mainFileShort+'" -Parameters "/VERYSILENT /NORESTART /LOG=`"$nsisLog`"" -WindowStyle Hidden'
  }
  
  
  if ($appDefObject.uninstall -eq $null -or $appDefObject.uninstall -eq '')
  {
    write-host 'No InnoSetup UnInstall parameter found. Generating script...'
    $appDefObject.uninstall=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Uninstalling '+($appDefObject.ProductName)+' '+($appDefObject.ProductVersion)+'. This may take some time. Please wait..."'
    $appDefObject.uninstall+=[Environment]::NewLine+'$uninstallApp=(Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {$_.DisplayName -like "*'+$appDefObject.ProductName+'*" -and $_.DisplayVersion -eq "'+$appDefObject.ProductVersion+'"}).UninstallString.replace("`"","")'
    $appDefObject.uninstall+=[Environment]::NewLine+'$nsisLog=join-path $configToolkitLogDir ($installName + "_" + $appDeployToolkitName + "_" + $deploymentType + "_NSIS.log")'
    $appDefObject.uninstall+=[Environment]::NewLine+'Execute-Process -Path "$uninstallApp" -Parameters "/VERYSILENT /NORESTART /LOG=$nsisLog" -WindowStyle Hidden'
  }
  
  
  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    write-host 'No InnoSetup detect parameter found. Generating script...'
    $appDefObject.detect="Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {`$_.DisplayName -like `"*"+$appDefObject.ProductName+"*`" -and `$_.DisplayVersion -eq `""+$appDefObject.ProductVersion+"`"}"
  }


  create-CMApplication -appDefObject $appDefObject -globalDefObject $globalDefObject
}

function New-NSISSetup {
  param
  (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  
  $mainFile=$appDefObject.mainFile
  


  if ($appDefObject.install -eq $null -or $appDefObject.install -eq '')
  {
    write-host 'No NSIS install parameter found. Generating script...'
    $appDefObject.install=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Installing '+($appDefObject.ProductName)+' '+($appDefObject.ProductVersion)+'. This may take some time. Please wait..."'
    $appDefObject.install+=[Environment]::NewLine+'Execute-Process -Path "'+$appDefObject.mainFileShort+'" -Parameters "/S" -WindowStyle Hidden'
  }
  
  if ($appDefObject.uninstall -eq $null -or $appDefObject.uninstall -eq '')
  {
    $appDefObject.uninstall=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Uninstalling '+($appDefObject.ProductName)+' '+($appDefObject.ProductVersion)+'. This may take some time. Please wait..."'
    if ($appDefObject.isX86) {
      $appDefObject.uninstall+=[Environment]::NewLine+'if ($is64Bit) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
    }
    else
    {
      $appDefObject.uninstall+=[Environment]::NewLine+'$regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"'
    }
    $appDefObject.uninstall+=[Environment]::NewLine+'$uninstallApp=(Get-ItemProperty $regPath | where {$_.DisplayName -like "*'+$appDefObject.ProductName+'*" -and $_.DisplayVersion -eq "'+$appDefObject.ProductVersion+'"}).UninstallString.replace("`"","")'
    $appDefObject.uninstall+=[Environment]::NewLine+'Execute-Process -Path "$uninstallApp" -Parameters "/S" -WindowStyle Hidden'
  }
  
  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    if ($appDefObject.isX86) {
      $appDefObject.detect='if ([IntPtr]::Size -eq 8) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
    }
    else
    {
      $appDefObject.detect='$regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"'
    }
    $appDefObject.detect+=[Environment]::NewLine+'Get-ItemProperty $regPath | where {$_.DisplayName -like "*'+$appDefObject.ProductName+'*" -and $_.DisplayVersion -eq "'+$appDefObject.ProductVersion+'"}'
  }
   
  create-CMApplication -appDefObject $appDefObject -globalDefObject $globalDefObject 
}

function check-mainfile{
  param
  (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  $realHash=Get-FileHash -Path $appDefObject.mainFile -Algorithm $appdefObject.hashType
  $realHash.hash -ieq $appDefObject.hashValue
}

[xml]$xml=Get-Content $appDefinition

$sourceFiles=(get-item $appdefinition).DirectoryName
$globalDef=get-GlobaleDefinition

$appDef=get-AppDefinition -xml $xml -sourcefiles $sourceFiles -globalDefObject $globalDef

write-host 'Globale Definition:'
$globalDef
#write-host 'Application Definition:'
#$appDef 

if ((check-mainfile -appDefObject $appDef -globalDefObject $globalDef) -or $ignoreHashError) {
    if (-not $ignoreHashError) {
        write-host -BackgroundColor Green -ForegroundColor Black "Hash value is correct. Creating application."
    }
    if ($xml.appdefinition.info.setupType -eq 'InnoSetup') {
        new-InnoSetup -appDefObject $appDef -globalDefObject $globalDef
    }
    elseif ($xml.appdefinition.info.setupType -eq 'NSIS') {
        new-NSISSetup -appDefObject $appDef -globalDefObject $globalDef
    }
    elseif ($xml.appdefinition.info.setupType -eq 'MSI') {
        new-MSISetup -appDefObject $appDef -globalDefObject $globalDef
    }
    elseif ($xml.appdefinition.info.setupType -eq 'MSP') {
        new-MSPSetup -appDefObject $appDef -globalDefObject $globalDef
    }
    write-host 'Resulting Application Definition:'
    $appDef
}
else
{
  write-host -BackgroundColor Red ("Error: Hash value of main file " + $appdef.mainfile + " doesn't match with value in XML file "+$appDefinition+" (you can ignore this error with the switch -ignoreHashError)")
}


