Set-StrictMode -Version latest

#marker for appdeploy file
$INSTALL_HASH='## <Perform Installation tasks here>'
$UNINSTALL_HASH='## <Perform Pre-Uninstallation tasks here>'
$beforeLocation=''


function Import-SCCMmoduleCert
{
<#
.Synopsis
   Imports the signed certificate used in the SCCM module into the certificate store on the local computer
.DESCRIPTION
   Requires administrative privileges to run. Run the function as the user that will have the cert
   imported. The function accepts no parameters and does not return any output.
.EXAMPLE
   Import-SCCMmoduleCert
.NOTES
    Created by Tore Groneng @ToreGroneng Tore.Groneng@gmail.com
    http://www.dexterposh.com/2014/06/powershell-sccm-2012-getting-started.html
#>
[CmdletBinding()]
[OutputType([int])]
Param()

    Write-verbose "Start $($MyInvocation.MyCommand.Name)"
    $sccmModulePath = "$(Split-Path $env:SMS_ADMIN_UI_PATH -Parent)\ConfigurationManager.psd1"

    Write-Verbose "Module path is $sccmModulePath, getting cert"
    $cert = Get-AuthenticodeSignature -FilePath "$sccmModulePath" -ErrorAction SilentlyContinue

    Write-Verbose "Creating a store object for LocalMachine\TrustedPublisher"
    $store = new-object System.Security.Cryptography.X509Certificates.X509Store("TrustedPublisher","LocalMachine")
    $store.Open("MaxAllowed")

    Write-Verbose "Adding cert to store"
    $store.Add($cert.SignerCertificate)
    $store.Close()

    Write-Verbose "Done"
} 

function initialize{
  [CmdletBinding()]
  param(
    [Parameter(Mandatory=$true)]
    [string]$Site,
    [Parameter(Mandatory=$true)]
    [string]$Siteserver
  )
  #Load the ConfigurationManager Module
  Import-SCCMmoduleCert

  #from http://www.dexterposh.com/2014/06/powershell-sccm-2012-getting-started.html
  if ((get-module ConfigurationManager) -eq $null) {
  try
  {
    $ConfigMgrModule = "$($env:SMS_ADMIN_UI_PATH | Split-Path -Parent)\ConfigurationManager.psd1"
    if (!(Test-Path -Path $ConfigMgrModule)) {
      throw 'Configuration Manager module not found in admin console path'
    }
    write-host 'Loading ConfigMgr Module from ' $ConfigMgrModule
    Import-Module "$ConfigMgrModule"
    #Get-Module
    $BeforeLocation = (Get-Location).Path
  } catch 
  {
    Write-Error $_.Exception.Message        
  }
  }
  $PSDefaultParameterValues =@{"Get-wmiobject:namespace"=("Root\SMS\site_"+$site);"Get-WMIObject:computername"=$siteServer}
  Set-Location $site":"
}

function replace-PathVariables {
  [CmdletBinding()]
  param(
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [string]$variable
  )
  $rt=$variable
  $appDefObject | Get-Member -MemberType NoteProperty | ForEach-Object {
    $name=$_.name
    $replace='{'+$_.name +'}'
    $rt=$rt.Replace($replace,$appDefObject."$name")
  }
  $rt
}

function New-SoftwareCollection {
  [CmdletBinding()]
  param(
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  
  initialize -Site $globalDefObject.site -Siteserver $globalDefObject.Siteserver
  
  $PSDefaultParameterValues =@{
        'get-cimclass:namespace'='Root\SMS\site_'+$globalDefObject.site;
        'get-cimclass:computername'=$globalDefObject.siteserver;
        'get-cimInstance:computername'=$globalDefObject.siteserver;
        'get-ciminstance:namespace'='Root\SMS\site_'+$globalDefObject.site;
        'get-wmiobject:namespace'='Root\SMS\site_'+$globalDefObject.site;
        'get-WMIObject:computername'=$globalDefObject.siteserver}
  
  $rootDeviceFolder=Get-CimInstance -ClassName SMS_ObjectContainerNode -Filter 'ObjectType=5000 and ParentContainerNodeid=0'
  
  $folderpath=join-path $globalDefObject.rootFolder (replace-PathVariables -variable $globalDefObject.collectionPathTemplate -appDefObject $appDefObject)
  write-debug ('new-SoftwareCollection: Joined Path: '+$folderpath)
  #create device collection folders=5000
  $targetFolder=new-FolderPath -folderPath $folderpath -parentID 0 -folderTyp 5000 -globalDefObject $globalDefObject
  #create application folders=6000
  $appFolder=new-FolderPath  -folderPath $appDefObject.Manufacturer -parentID 0 -folderTyp 6000 -globalDefObject $globalDefObject
  
  
  $colName=replace-PathVariables -variable $globalDefObject.collectionNameTemplate -appDefObject $appDefObject
  write-debug ('New-SoftwareCollection: '+$colName)
  $collection=Get-CMCollection -name $colName -CollectionType Device
  if (-not $collection) {
    $collection=New-CMCollection -CollectionType Device -Name $colName -Comment ('Software Deployment Target for '+$appDefObject.ProductName) -LimitingCollectionName $globalDefObject.limitingCollection  -RefreshType Periodic -RefreshSchedule (New-CMSchedule -Start (get-date) -RecurInterval Minutes -RecurCount $globalDefObject.updateCollectionMin) 
    write-host ("Collection $colName successfull created. ID:"+$collection.CollectionID)
    if ($globalDefObject.includeCollection -ne $null) {
    #MB Probleme mit Alias... originalfunktion nehmen
#      Add-CMDeviceCollectionIncludeMembershipRule -CollectionID $collection.CollectionID -IncludeCollectionName $globalDefObject.includeCollection
        $ruleClassName = "SMS_CollectionRuleIncludeCollection"
        $searchCriteria = New-Object -TypeName ([Microsoft.ConfigurationManagement.PowerShell.Provider.SmsProviderSearch])
        $searchCriteria.Add("Name", $globalDefObject.includeCollection)
        Add-CMCollectionMembershipRule -SearchCriteria $searchCriteria -RuleClassName $ruleClassName -CollectionType Device -RulePropertyName "IncludeCollectionID" -CollectionID $collection.CollectionID
    }
  }
  else
  {
    write-host -BackgroundColor Yellow -ForegroundColor Black ('Warning: Collection {0} already exist with ID {1}. Will not create a new one.' -f $colName,$collection.CollectionID)
  }
  Move-CMObject -FolderPath ($globalDefObject.site+':\DeviceCollection\'+$folderpath) -InputObject $collection
  
  $collection=Get-CMCollection -CollectionType Device -name $colName
  $collection
  Set-Location $BeforeLocation
}

#device collection folder=5000, user collection=5001, Application folder=6000
function Get-Folder {
  param
  (
    [Parameter(Mandatory=$true)]
    [System.String]
    $name,
    [Parameter(Mandatory=$true)]
    [System.Int32]
    $parentID,
    [Parameter(Mandatory=$false)]
    [System.Int32]
    $folderTyp = 5000
  )

  initialize -Site $globalDefObject.site -Siteserver $globalDefObject.Siteserver

  Get-CimInstance -ClassName SMS_ObjectContainerNode -Filter "ObjectType=$folderTyp and ParentContainerNodeid=$parentID and Name='$name'"
}
 
 #device collection folder=5000, user collection=5001, Application folder=6000
 function New-Folder
{
  param
  (
    [Parameter(Mandatory=$true)]
    [System.String]
    $name,
    [Parameter(Mandatory=$true)]
    [System.Int32]
    $parentID,
    [Parameter(Mandatory=$false)]
    [System.Int32]
    $folderTyp = 5000,
    [Parameter(Mandatory=$true)]
    [System.Management.Automation.PSObject]
    $globalDefObject
  )
  
  initialize -Site $globalDefObject.site -Siteserver $globalDefObject.Siteserver

  $rt=get-folder $name $parentID $folderTyp
  if (-not $rt) {
    $rt=New-CimInstance -ClassName SMS_ObjectContainerNode -Property @{Name=$name;ObjectType=$folderTyp;ParentContainerNodeid=$parentID;SourceSite=$globalDefObject.site} -Namespace $globalDefObject.namespace -ComputerName $globalDefObject.siteserver 
    Write-Host "Folder $name successfull created."
  }
  else
  {
    write-verbose "Folder $name already exists."
  }
  $rt
}
   #device collection folder=5000, user collection=5001, Application folder=6000
 function New-FolderPath
{
  param
  (
    [Parameter(Mandatory=$true)]
    [System.String]
    $folderPath,
    [Parameter(Mandatory=$true)]
    [System.Int32]
    $parentID,
    [Parameter(Mandatory=$false)]
    [System.Int32]
    $folderTyp = 5000,
    [Parameter(Mandatory=$true)]
    [System.Management.Automation.PSObject]
    $globalDefObject
  )
  
  $folders=$folderPath.split('\')
  $parentID=0
  $folders | ForEach-Object {
    $folder=$_
    $rt=new-Folder -name $folder -parentID $parentID -folderTyp $folderTyp -globalDefObject $globalDefObject
    $parentID=$rt.ContainerNodeID
  }
  $rt
}

function get-DeploymentTypeByHash{
[CmdletBinding()]
  param(
    [Parameter(Mandatory=$true)]
    [string]$hashType,
    [Parameter(Mandatory=$true)]
    [string]$hashValue)

    initialize -Site $globalDefObject.site -Siteserver $globalDefObject.Siteserver

    Get-CMApplication | Get-CMDeploymentType | where {$_.LocalizedDescription -like ('*'+$hashType+':'+$hashValue)}

}

#source: http://www.dexterposh.com/2014/06/powershell-sccm-2012-create-dependency.html
function add-dependency {
[CmdletBinding()]
  param(
    [Parameter(Mandatory=$true)]
    [object]$dependentDT,
    [Parameter(Mandatory=$true)]
    [object]$newAppDT,
    [Parameter(Mandatory=$true)]
    [string]$DependencyName, 
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject,   
    [bool]$autoinstall=$false
)

  initialize -Site $globalDefObject.site -Siteserver $globalDefObject.Siteserver


#load the Application Management DLL
Add-Type -Path "$(Split-Path $Env:SMS_ADMIN_UI_PATH)\Microsoft.ConfigurationManagement.ApplicationManagement.dll"
Add-Type -Path "$(Split-Path $Env:SMS_ADMIN_UI_PATH)\Microsoft.ConfigurationManagement.ApplicationManagement.MsiInstaller.dll"

  $PSDefaultParameterValues =@{"Get-wmiobject:namespace"=("Root\SMS\site_"+$globalDefObject.site);"Get-WMIObject:computername"=$globalDefObject.siteServer}
 #Creating Type Accelerators - for making assemlby refrences easier later
$accelerators = [PSObject].Assembly.GetType('System.Management.Automation.TypeAccelerators')
$accelerators::Add('SccmSerializer',[type]'Microsoft.ConfigurationManagement.ApplicationManagement.Serialization.SccmSerializer')


#region Application 1
#get direct reference to the Application's WMI Instance
#$application1 = [wmi](Get-WmiObject -Query "select * from sms_application where LocalizedDisplayName='$ApplicationName1' AND ISLatest='true'").__PATH
$newApp = [wmi](Get-WmiObject -Query ("select * from sms_application where ModelName='"+$newAppDT.AppModelName+"' and ISLatest='true'")).__PATH


#Deserialize the SDMPackageXML
$App1Deserializedstuff = [SccmSerializer]::DeserializeFromString($newApp.SDMPackageXML)

#endregion Application 1


#region Application 2

#Name of the Application which will be added as a dependency
#$ApplicationName2 = ".NET4"

#Reference to the above application
#$application2 = [wmi](Get-WmiObject -Query "select * from sms_application where LocalizedDisplayName='$ApplicationName2' AND ISLatest='true'").__PATH
#$application2 = [wmi](Get-WmiObject -Query ("select * from sms_application where ModelName='"+$dependentDT.AppModelName+"' and ISLatest='true'")).__PATH

#deserialize the XML
#$App2Deserializedstuff = [SccmSerializer]::DeserializeFromString($application2.SDMPackageXML)

#endregion 

#region 1

#Store the arguments before hand
$ApplicationAuthoringScopeId = ($dependentDT.AppModelName -split "/")[0]
$ApplicationLogicalName = ($dependentDT.AppModelName -split "/")[1]
#$ApplicationVersion =  $application2.SourceCIVersion
$ApplicationVersion = 0
$DeploymentTypeAuthoringScopeId = ($dependentDT.ModelName -split "/")[0]
$DeploymentTypeLogicalName = ($dependentDT.ModelName -split "/")[1]
#$DeploymentTypeVersion = $App2Deserializedstuff.DeploymentTypes.Version
$DeploymentTypeVersion=0 #um immer die neueste revision zu nutzen
$EnforceDesiredState = $autoinstall  #this determines if the dependency Application will be auto installed

#endregion 1

#region 2
# set the Desired State as "Required" or Mandatory
$DTDesiredState = [Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Expressions.DeploymentTypeDesiredState]::Required

#endregion 2

#region 3
#create the intent expression which will be addded to the Operand
$intentExpression = new-object Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Expressions.DeploymentTypeIntentExpression -ArgumentList  $ApplicationAuthoringScopeId, $ApplicationLogicalName, $ApplicationVersion, $DeploymentTypeAuthoringScopeId, $DeploymentTypeLogicalName, $DeploymentTypeVersion, $DTDesiredState, $AutoInstall

#Create the Operand - Note the typename of this one
$operand = New-Object  Microsoft.ConfigurationManagement.DesiredConfigurationManagement.CustomCollection[Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Expressions.DeploymentTypeIntentExpression]

#add the Intent Expression to the Operand
$operand.Add($intentExpression)

#endregion 3

 #region 4

#create the new OR operator
$OrOperator = [Microsoft.ConfigurationManagement.DesiredConfigurationManagement.ExpressionOperators.ExpressionOperator]::Or

#endregion 4 

 #region 5

#Now the Operator and Operand are added to the Expression
$BaseExpression = New-Object -TypeName Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Expressions.DeploymentTypeExpression -ArgumentList $OrOperator,$operand

#endregion 5 

# Create the Severity Critical
$severity = [Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Rules.NoncomplianceSeverity]::Critical

# Create the Empty Rule Context
$RuleContext = New-Object -TypeName Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Rules.RuleScope

#Create the Annotation - Name & description of the Dependency
$annotation = New-Object -TypeName Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Rules.Annotation
$annotation.DisplayName.Text = $dependencyName

#Create the new DeploymentType Rule
$DTRUle = New-Object -TypeName Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Rules.DeploymentTypeRule -ArgumentList $("DTRule_"+[guid]::NewGuid().Guid),$severity, $annotation, $BaseExpression


#add the DepolymentType Rule to Dependecies to the correct dt
$App1Deserializedstuff.DeploymentTypes.Keys | ForEach-Object {
   if (($_.scope+"/"+$_.name) -eq $newAppDT.modelname) {
    $App1Deserializedstuff.DeploymentTypes[$_].Dependencies.add($DTRule)
   }
}
#ORIG: $App1Deserializedstuff.DeploymentTypes[0].Dependencies.Add($DTRUle)

# Serialize the XML
$newappxml = [SccmSerializer]::Serialize($App1Deserializedstuff, $false)

#set the property back on the local copy of the Object
$newApp.SDMPackageXML = $newappxml

#Now time to set the changes back to the ConfigMgr
$newApp.Put() 

}

#source: http://www.dexterposh.com/2014/05/powershell-sccm-2012-create.html
# Usage:
# $app=get-DeploymentTypeByHash -hashType SHA256 -hashValue 8371F27009A9286CAC0D9812E0411DDBFCB106456E46FFEC50E64054E75E4632
#$newApp=Get-CMApplication -Name "Oracle Corporation Java 8 Update 71 8.0.710.15 x86"
#add-Supersedence -supersedenceDT $app -newApp $newApp -uninstall $true
function add-Supersedence{
[CmdletBinding()]
  param(
    [Parameter(Mandatory=$true)]
    [object]$supersedenceDT,
    [Parameter(Mandatory=$true)]
    [object]$newAppDT,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject,
    [bool]$uninstall=$false
)
 
   initialize -Site $globalDefObject.site -Siteserver $globalDefObject.Siteserver

#load the Application Management DLL
Add-Type -Path "$(Split-Path $Env:SMS_ADMIN_UI_PATH)\Microsoft.ConfigurationManagement.ApplicationManagement.dll"

  $PSDefaultParameterValues =@{"Get-wmiobject:namespace"=("Root\SMS\site_"+$globalDefObject.site);"Get-WMIObject:computername"=$globalDefObject.siteServer}

$newApp = [wmi](Get-WmiObject -Query ("select * from sms_application where ModelName='"+$newAppDT.AppModelName+"' and ISLatest='true'")).__PATH

#write-host $newApp
#Reference to the above application
#write-host ("SDT: "+$supersedenceDT)
#write-host ("SDT AppModelName: "+$supersedenceDT.AppModelName)
$supersededapplication = [wmi](Get-WmiObject -Query ("select * from sms_application where ModelName='"+$supersedenceDT.AppModelName+"' and ISLatest='true'")).__PATH
#write-host ("Sda: "+$supersededapplication)
#$supersededapplication | Get-Member  | foreach-object {write-host ($_.name+' '+$_.Definition)}
#deserialize the XML
$supersededDeserializedstuff = [Microsoft.ConfigurationManagement.ApplicationManagement.Serialization.SccmSerializer]::DeserializeFromString($supersededapplication.SDMPackageXML)

# set the Desired State for the Superseded Application's Deployment type to "prohibit" from running
$DTDesiredState = [Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Expressions.DeploymentTypeDesiredState]::Prohibited 

 #Store the arguments before hand
$ApplicationAuthoringScopeId = ($supersededapplication.CI_UniqueID -split "/")[0]
$ApplicationLogicalName = ($supersededapplication.CI_UniqueID -split "/")[1]
$ApplicationVersion =  $supersededapplication.SourceCIVersion
#DT: CI_UniqueID                        : ScopeId_491CCB69-995E-4E8F-A09D-5275EAA6D546/DeploymentType_987db291-7cfa-4d4f-a2e3-462814df0884/1
$DeploymentTypeAuthoringScopeId = ($supersedenceDT.CI_UniqueID -split "/")[0]
$DeploymentTypeLogicalName = ($supersedenceDT.CI_UniqueID -split "/")[1]
$DeploymentTypeVersion = ($supersedenceDT.CI_UniqueID -split "/")[2]

#create the intent expression
$intentExpression = new-object Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Expressions.DeploymentTypeIntentExpression -ArgumentList  $ApplicationAuthoringScopeId, $ApplicationLogicalName, $ApplicationVersion, $DeploymentTypeAuthoringScopeId, $DeploymentTypeLogicalName, $DeploymentTypeVersion, $DTDesiredState, $uninstall

# Create the Severity None
$severity = [Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Rules.NoncomplianceSeverity]::None

# Create the Empty Rule Context
$RuleContext = New-Object -TypeName Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Rules.RuleScope


#Create the new DeploymentType Rule
$DTRUle = New-Object -TypeName Microsoft.SystemsManagementServer.DesiredConfigurationManagement.Rules.DeploymentTypeRule -ArgumentList $severity, $null, $intentExpression


#Deserialize the SDMPackageXML
$Deserializedstuff = [Microsoft.ConfigurationManagement.ApplicationManagement.Serialization.SccmSerializer]::DeserializeFromString($newApp.SDMPackageXML)
#$Deserializedstuff = [Microsoft.ConfigurationManagement.ApplicationManagement.Serialization.SccmSerializer]::DeserializeFromString(([wmi](Get-WmiObject -Query ("select * from sms_application where ModelName='"+$newApp.ModelName+"' and ISLatest='true'")).__PATH).SDMPackageXML)

#add the supersedence to the deployment type
#workaround a bug... if superseded array is 0 put it  two times...


$dtfound=$false 
$Deserializedstuff.DeploymentTypes.Keys | ForEach-Object {
 if (($_.scope+"/"+$_.name) -eq $newAppDT.modelname) {
    $Deserializedstuff.DeploymentTypes[$_].Supersedes.add($DTRule)
    $dtFound=$true 
   }
}
#orig $Deserializedstuff.DeploymentTypes[0].Supersedes.Add($DTRUle) 
if ($dtfound) {
# Serialize the XML
$newappxml = [Microsoft.ConfigurationManagement.ApplicationManagement.Serialization.SccmSerializer]::Serialize($Deserializedstuff, $true)

#set the property back on the local copy of the Object
$newApp.SDMPackageXML = $newappxml

#Now time to set the changes back to the ConfigMgr
$newApp.Put()
}
else
{
    write-host "ERROR: DT not found."
}
}

function New-CMApplicationDetect{
  [CmdletBinding()]
  param(
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  
  initialize -Site $globalDefObject.site -Siteserver $globalDefObject.siteserver
  

  $appName=replace-PathVariables -appDefObject $appDefObject -variable $globalDefObject.applicationNameTemplate
  $depName=replace-PathVariables -appDefObject $appDefObject -variable $globalDefObject.deploymentTypeTemplate
  $app=Get-CMApplication -Name $appName
  if (-not $app) {
    $app=New-CMApplication -Name $appName -Description 'Created by automateAppDeployment from mbaeker.de' -SoftwareVersion $appDefObject.ProductVersion -AutoInstall $true -Publisher $appDefObject.Manufacturer
  }
  else
  {
    write-host -BackgroundColor Yellow -ForegroundColor Black ('Warning: Application {0} already exist. Will not create a new one.' -f $appName)

  }
  
  
  
  #    Parameter Set: AddDeploymentTypeByScriptInstallerManual (http://technet.microsoft.com/en-us/library/jj870953%28v=sc.20%29.aspx)
  #Add-CMDeploymentType -ApplicationName <String> -DeploymentTypeName <String> -DetectDeploymentTypeByCustomScript -InstallationProgram <String> -ManualSpecifyDeploymentType -ScriptContent <String> -ScriptInstaller -ScriptType <ScriptLanguage> [-AdministratorComment <String> ] [-AllowClientsToShareContentOnSameSubnet <Boolean> ] [-ContentLocation <String> ] [-EstimatedInstallationTimeMinutes <Int32> ] [-InstallationBehaviorType <InstallationBehaviorType> {InstallForSystem | InstallForSystemIfResourceIsDeviceOtherwiseInstallForUser | InstallForUser} ] [-InstallationProgramVisibility <UserInteractionMode> {Normal | Minimized | Maximized | Hidden} ] [-InstallationStartIn <String> ] [-Language <String[]> ] [-LogonRequirementType <LogonRequirementType> {OnlyWhenNoUserLoggedOn | OnlyWhenUserLoggedOn | WhereOrNotUserLoggedOn | WhetherOrNotUserLoggedOn} ] [-MaximumAllowedRunTimeMinutes <Int32> ] [-PersistContentInClientCache <Boolean> ] [-RequiresUserInteraction <Boolean> ] [-RunInstallationProgramAs32BitProcessOn64BitClient <Boolean> ] [-RunScriptAs32bitProcessOn64bitClient <Boolean> ] [-UninstallProgram <String> ] [-UninstallStartIn <String> ] [-Confirm] [-WhatIf] [ <CommonParameters>]
  
  $dep=get-CMDeploymentType -ApplicationName $app.LocalizedDisplayName -DeploymentTypeName $depName
  if (-not $dep) {
    #source http://www.dexterposh.com/2014/04/powershell-sccm-2012-r2-deploy.html
    #write-host $script
    
    
    $DeploymentTypeHash = @{
                    DeploymentTypeName = $depName   #Name given to the Deployment Type
                    InstallCommand ='deploy-application.exe'  #Command line to Run for install
                    ApplicationName = $appName #Application Name 
                    ScriptLanguage = 'PowerShell'
                    ScriptText = $appDefObject.detect
                    ContentLocation = $appDefObject.uncTargetFiles
                    RequiresUserInteraction = $false  #Don't let User interact with this
                    LogonRequirementType = 'WhereOrNotUserLoggedOn'
                    UserInteractionMode = 'Normal'  # Hide the PowerShell Console
                    UninstallCommand  ='deploy-Application.EXE -DeploymentType "Uninstall"' #Command line to Run for un-Install
                    Comment = ('Created by automateAppDeployment from mbaeker.de '+"`n"+$appDefObject.hashType+':'+$appDefObject.hashValue)
                   # ScriptInstaller = $true
                   # ManualSpecifyDeploymentType=$true #deprecated
                   # DetectDeploymentTypeByCustomScript = $true # Yes deployment type will use a custom script to detect the presence of this 
                    #InstallationFileLocation = $appDefObject.uncTargetFiles
                    #InstallationBehaviorType = 'InstallForSystem' # Targeting Devices here
                    
                    }
    if (-not $appDefObject.hasUninstall) {
      $DeploymentTypeHash.remove('UninstallProgram')
    }                           

#    Add-CMDeploymentType @DeploymentTypeHash 
     Add-CMScriptDeploymentType @DeploymentTypeHash | out-null
  }
  else
  {
    write-host -BackgroundColor Yellow -ForegroundColor Black ('Warning: DeploymentType {0} already exist. Will not create a new one.' -f $depName)
  }
  
  #region check supersedence
  $appDefObject.supersedence.keys | ForEach-Object {
    $sKey=$_
    $hashType=$_.split(":")[0]
    $hashValue=$_.split(":")[1]
    write-host ("Searching for superseded application "+$_)
    $sApps=get-DeploymentTypeByHash -hashType $hashType -hashValue $hashValue
    if ($sApps) {
        #sonderfall, falls mehrere applications mit dem hash gefunden werden...
       $sApps | ForEach-Object {
        $sApp=$_
        write-host ("Adding "+$sApp.LocalizedDisplayName+" to superseded applications.")
            #jedes mal neu holen, da durch dependency neuere version vorhanden ist
        $appDT=Get-CMDeploymentType -ApplicationName $app.LocalizedDisplayName -DeploymentTypeName $depName
        $rt=add-Supersedence -supersedenceDT $sApp -newAppDT $appDT -uninstall $appDefObject.supersedence[$sKey] -globalDefObject $globalDefObject
        #app has changed. reload it
        $app=Get-CMApplication -Name $appName
        }
    }
    else
    {
        write-host ("Couldn't find superseded application with hash "+$_)
    }
  }
  #endregion

  #region check dependency
  $appDefObject.dependency.keys | ForEach-Object {
    $depGroup=$_
    [hashtable]$apps=$appDefObject.dependency[$_]
    $apps.keys | ForEach-Object {
        $hashType=$_.split(":")[0]
        $hashValue=$_.split(":")[1]
        write-host ("Searching for dependent application "+$_+" in group "+$DepGroup)
        $sApp=get-DeploymentTypeByHash -hashType $hashType -hashValue $hashValue
        if ($sApp) {
            write-host ("Adding "+$sApp.LocalizedDisplayName+" to dependent applications.")
            #jedes mal neu holen, da durch dependency neuere version vorhanden ist
            $appDT=Get-CMDeploymentType -ApplicationName $app.LocalizedDisplayName -DeploymentTypeName $depName
            $rt=add-dependency -dependentDT $sApp -newAppDT $appDT -autoinstall $apps[$_] -DependencyName $depGroup -globalDefObject $globalDefObject
            #app has changed. reload it
            $app=Get-CMApplication -Name $appName
        }
        else
        {
            write-host ("Couldn't find superseded application with hash "+$_)
        }
     }
  }
  #endregion
  #$app=Get-CMApplication -Name $appName
  #$app
  #cd $BeforeLocation
}
 
function Move-CMApp{
  [CmdletBinding()]
  param(
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  ) 

    initialize -Site $globalDefObject.site -Siteserver $globalDefObject.Siteserver

  Move-CMObject -FolderPath ($globalDefObject.site+':\Application\'+$appDefObject.Manufacturer) -ObjectId $appDefObject.applicationObjectID
}

function Start-CMDeployment{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
  [PSCustomObject]$globalDefObject)

  initialize -Site $globalDefObject.site -Siteserver $globalDefObject.Siteserver
  $PSDefaultParameterValues =@{"Get-wmiobject:namespace"=('Root\SMS\site_{0}' -f $globalDefObject.site);"Get-WMIObject:computername"=$globalDefObject.Siteserver}
  
  $colName=replace-PathVariables -appDefObject $appDefObject -variable $globalDefObject.collectionNameTemplate
  $appName=replace-PathVariables -appDefObject $appDefObject -variable $globalDefObject.applicationNameTemplate
  
  
  #search if content is already distributed to group
  $dpGroupID=Get-WmiObject -Class SMS_DistributionPointGroup -Filter ('Name="{0}"' -f $globalDefObject.DistributionPointGroupName) 
  if ($dpGroupID -eq $null) {
    throw ('Distribution Point Group {0} doesn''t exists' -f $globalDefObject.DistributionPointGroupName)
  }
  write-debug ('Start-CMDeployment: DP Group exists: {0}' -f $dpGroupID.groupID)
  $ContentDist=Get-WmiObject -class SMS_DPGroupDistributionStatusDetails -Filter "ObjectType = 512 AND MessageState = 1 OR MessageState = 2" | where {$_.ContentName -eq $appName -and $_.groupID -eq $dpGroupID.groupID}

  if ($ContentDist -eq $null) {
    Start-CMContentDistribution -ApplicationName ($appName) -DistributionPointGroupName ($globalDefObject.DistributionPointGroupName) 
  }
  else
  {
    write-host -BackgroundColor Yellow -ForegroundColor Black ('Warning: Content of {0} already distributed to {1}. Will not resend.' -f $appName,$globalDefObject.DistributionPointGroupName)
  }

  if ((Get-CMApplicationDeployment -Name ($appName) -CollectionName ($colName)) -eq $null) {
    New-CMApplicationDeployment -CollectionName ($colName) -Name ($appname) -DeployAction Install -DeployPurpose Available -UserNotification DisplayAll -AvailableDateTime (get-date) -TimeBaseOn LocalTime  
  }
   else
   {
    write-host -BackgroundColor Yellow -ForegroundColor Black ('Warning: Application deployment {0} already deployed to {1}. Will not create a new one.' -f $appName,$colName)
   }
}


#region Function Get-MsiTableProperty
Function Get-MsiTableProperty {
<#
.SYNOPSIS
	Get all of the properties from an MSI table and return as a custom object.
.DESCRIPTION
	Use the Windows Installer object to read all of the properties from a MSI table.
.PARAMETER Path
	The fully qualified path to an MSI file.
.PARAMETER Table
	The name of the the MSI table from which all of the properties must be retrieved. Default is: 'Property'.
.PARAMETER ContinueOnError
	Continue if an error is encountered. Default is: $true.
.EXAMPLE
	Get-MsiTableProperty -Path 'C:\Package\AppDeploy.msi'
	Retrieve all of the properties from the default 'Property' table.
.EXAMPLE
	Get-MsiTableProperty -Path 'C:\Package\AppDeploy.msi' -Table 'Property' | Select-Object -ExpandProperty ProductCode
	Retrieve all of the properties from the 'Property' table and then pipe to Select-Object to select the ProductCode property.
.NOTES
	This is an internal script function and should typically not be called directly.
.LINK
	http://psappdeploytoolkit.codeplex.com
#>
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$true)]
    [ValidateScript({ $_ | Test-Path -PathType Leaf })]
    [string]$Path,
    [Parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [string]$Table = 'Property',
    [Parameter(Mandatory=$false)]
    [ValidateNotNullorEmpty()]
    [boolean]$ContinueOnError = $true
  )
  
  Begin {
    ## Get the name of this function and write header
    [string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
    #Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
    
    [scriptblock]$InvokeMethod = {
      Param (
        [__comobject]$Object,
        [string]$MethodName,
        [object[]]$ArgumentList
      )
      Write-Output $Object.GetType().InvokeMember($MethodName, [System.Reflection.BindingFlags]::InvokeMethod, $null, $Object, $ArgumentList, $null, $null, $null)
    }
    
    [scriptblock]$GetProperty = {
      Param (
        [__comobject]$Object,
        [string]$PropertyName,
        [object[]]$ArgumentList
      )
      Write-Output $Object.GetType().InvokeMember($PropertyName, [System.Reflection.BindingFlags]::GetProperty, $null, $Object, $ArgumentList, $null, $null, $null)
    }
  }
  Process {
    Try {
      #Write-Log -Message "Get properties from MSI file [$Path] in table [$Table]" -Source ${CmdletName}
      
      ## Create an empty object to store properties in
      [psobject]$TableProperties = New-Object -TypeName PSObject
      ## Create a Windows Installer object
      [__comobject]$Installer = New-Object -ComObject WindowsInstaller.Installer -ErrorAction 'Stop'
      ## Open MSI database in read only mode
      [int32]$OpenMSIReadOnly = 0
      [__comobject]$Database = &$InvokeMethod -Object $Installer -MethodName 'OpenDatabase' -ArgumentList @($Path, $OpenMSIReadOnly)
      ## Open the "Property" table view
      [__comobject]$View = &$InvokeMethod -Object $Database -MethodName 'OpenView' -ArgumentList @("SELECT * FROM $Table")
      &$InvokeMethod -Object $View -MethodName 'Execute' | Out-Null
      
      ## Retrieve the first row from the "Properties" table
      [__comobject]$Record = &$InvokeMethod -Object $View -MethodName 'Fetch'
      ## If the first row was successfully retrieved, then save data and loop through the entire table
      While ($Record) {
        #  Add property and value to custom object
        $TableProperties | Add-Member -MemberType NoteProperty -Name (&$GetProperty -Object $Record -PropertyName 'StringData' -ArgumentList @(1)) -Value (&$GetProperty -Object $Record -PropertyName 'StringData' -ArgumentList @(2))
        #  Retrieve the next row in the table
        [__comobject]$Record = &$InvokeMethod -Object $View -MethodName 'Fetch'
      }
      
      Write-Output $TableProperties
    }
    Catch {
      #Write-Log -Message "Failed to get the MSI table [$Table]. `n" -Severity 3 -Source ${CmdletName}
      If (-not $ContinueOnError) {
        Throw "Failed to get the MSI table [$Table]: $($_.Exception.Message)"
      }
    }
    Finally {
      If ($View) {
        &$InvokeMethod -Object $View -MethodName 'Close' -ArgumentList @() | Out-Null
      }
    }
  }
  End {
    #Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
  }
}
#endregion

#source: http://stackoverflow.com/questions/15662799/powershell-function-to-replace-or-add-lines-in-text-files
function setConfig {
  param
  (
    [Parameter(Mandatory=$true)]
    [System.Object]
    $file,
    [Parameter(Mandatory=$true)]
    [System.Object]
    $key,
    [Parameter(Mandatory=$true)]
    [System.Object]
    $value,
    [bool]
    $ignoreMissing=$false
  )
  
  $content = Get-Content $file
  $search='^[\t\s]*'+[regex]::escape($key)+'\s*='
  if ( $content -match $search ) {
    $content=$content -replace "$search.*", "$key = $value" |  set-content $file -force 
  } else
  {
    if ($ignoreMissing -eq $false) {
        write-host -ForegroundColor Yellow "Couldn't find and replace key $key in file $file"
    }
  }
}


function replaceLine {
  param
  (   
    [Parameter(Mandatory=$true)]
    [System.Object]
    $file,
    [Parameter(Mandatory=$true)]
    [System.Object]
    $line,
    [Parameter(Mandatory=$true)]
    [System.Object]
    $newline,
    [bool]
    $ignoreMissing=$false
  )
  
  $content = Get-Content $file
  $search='^[\t\s]*'+[regex]::escape($line)
  
  #workaround, damit $_ im regex replace nicht aufgelst wird...
  if ( $content -match $search ) {
    $newContent='' 
    $content | ForEach-Object { 
      if ($_ -match $search) {
        $newContent+=$newLine+[Environment]::NewLine
      }
      else
      {
        $newContent+=$_+[Environment]::NewLine
      }
    }
    
    $newcontent | set-content -Path $file -Force
  } else
  {
      if ($ignoreMissing -eq $false) {
        write-host -ForegroundColor Yellow "Couldn't find and replace line $line (search: $search) in file $file"
      }
  }
}



function New-ApplicationSource {
  [CmdletBinding()]
  param(
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )

    initialize -Site $globalDefObject.site -Siteserver $globalDefObject.Siteserver

  $appName=replace-PathVariables -appDefObject $appDefObject -variable $globalDefObject.applicationNameTemplate
  $targetPath=$appDefObject.targetFiles
  
  #create folder
  if (-not (Test-Path  ($targetpath)))
  {
    new-item -Path  ($targetpath) -ItemType Directory |out-null
  }
  else
  {
      write-host -BackgroundColor Yellow -ForegroundColor Black ('Warning: Filesystem path {0} already exist.' -f $targetPath)
  }
  
  Get-ChildItem -Path ($globalDefObject.appdeploySource) | ForEach-Object {
    # $_.FullName
    Copy-Item -Force -Path  ($_.FullName) -Recurse -Destination  ($targetPath)
  }
  
  #copy source files to files in destination
  $filesDir=Join-Path $targetPath 'Files'
  #does files folder exists?
  if (-not (Test-Path -PathType Container -Path $filesDir)) {
     new-item -path $filesDir -ItemType Directory
  }
  Get-ChildItem -Path $appDefObject.sourceFiles | ForEach-Object {
    Copy-Item -Force -Path ($_.FullName) -Recurse -Destination ($filesDir)
  }
  
  $deployAppFile=Join-Path $targetPath 'Deploy-Application.ps1'
  
  #setting base infos in Deploy-Application File
  setConfig $deployAppFile "[string]`$appVendor" ("`""+$appDefObject.Manufacturer+"`"")
  setConfig $deployAppFile "[string]`$appName" ("`""+$appDefObject.ProductName+"`"")
  setConfig $deployAppFile "[string]`$appVersion" ("`""+$appDefObject.ProductVersion+"`"")
  setConfig $deployAppFile "[string]`$appScriptDate" ("`""+(get-date)+"`"")
  setConfig $deployAppFile "[string]`$appScriptAuthor" ("`""+[Environment]::UserDomainName+'\'+[Environment]::UserName+"`"")
  
  #install
  replaceline $deployAppFile "Show-InstallationWelcome -CloseApps 'iexplore' -AllowDefer -DeferTimes 3 -CheckDiskSpace -PersistPrompt" "#Show-InstallationWelcome -CloseApps 'iexplore' -AllowDefer -DeferTimes 3 -CheckDiskSpace -PersistPrompt"
  $installCommand=''
  
  if ($appdefObject.install -eq $null -or $appDefObject.install -eq '') {
    write-host 'new-AppDeployApplication: no install parameter. Trying to add MSI Files'
    #install  for msi files
    Get-ChildItem -Path $filesdir -Filter *.msi | ForEach-Object {
      addAppDeployMSI $deployAppFile $_.Name
    }
  }
  else
  {
    replaceline $deployAppFile $INSTALL_HASH ($appDefObject.install+[Environment]::NewLine+$INSTALL_HASH)
  }
  
  if ( $appDefObject.uninstall -eq $null -or $appDefObject.uninstall -eq '') {
    write-host 'new-AppDeployApplication: no unInstallString parameter. Trying to add MSI Files'
    #uninstall for msi files
    Get-ChildItem -Path $filesdir -Filter *.msi | ForEach-Object {
      addAppDeployUnMSI $deployAppFile $_.Name
    }
  }
  else
  {
    replaceline $deployAppFile $UNINSTALL_HASH ($appDefObject.uninstall+[Environment]::NewLine+$UNINSTALL_HASH)
  }
  
  #AppToolkit 3.6: ' -> "
  replaceline $deployAppFile "Show-InstallationPrompt -Message `"You can customize text" "" $true
  #appToolkit 3.6.7
  replaceline $deployAppFile "If (-not `$useDefaultMsi) { Show-InstallationPrompt -Message 'You can customize text" "" $true
  #uninstall
  replaceline $deployAppFile "Show-InstallationWelcome -CloseApps 'iexplore' -CloseAppsCountdown 60" "#Show-InstallationWelcome -CloseApps 'iexplore' -CloseAppsCountdown 60" $true
}

function addAppDeployMSI {
  param
  (
    [Parameter(Mandatory=$true)]
    [System.String]
    $deployAppFile,
    [Parameter(Mandatory=$true)]
    [System.String]
    $msiFileName
  )
  
  $installCommand=[Environment]::NewLine+"Execute-MSI -Action Install -Path `""+$msiFileName+"`""
  replaceline $deployAppFile $INSTALL_HASH ($installCommand+[Environment]::NewLine+$INSTALL_HASH)
}

function addAppDeployUnMSI {
  param
  (
    [Parameter(Mandatory=$true)]
    [System.String]
    $deployAppFile,
    [Parameter(Mandatory=$true)]
    [System.String]
    $msiFileName
  )
  
  $installCommand=[Environment]::NewLine+"Execute-MSI -Action Uninstall -Path `""+$msiFileName+"`""
  replaceline $deployAppFile $UNINSTALL_HASH ($installCommand+[Environment]::NewLine+$UNINSTALL_HASH)
}

function create-CMApplication{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  ) 
    #create filesystem folders and appdeployment toolkit
  $rt=new-ApplicationSource  -appDefObject $appDefObject -globalDefObject $globalDefObject
  if ($globalDefObject.doConfigMgrIntegration) {
    #create new folder and collection in configmgr
    $collection=new-SoftwareCollection -appDefObject $appDefObject -globalDefObject $globalDefObject
    #create new application in configmgr
    $rt=new-CMApplicationDetect -appDefObject $appDefObject -globalDefObject $globalDefObject
    $app=Get-CMApplication -Name (replace-PathVariables -appDefObject $appDefObject -variable $globalDefObject.applicationNameTemplate)
    $appDefObject.applicationObjectID=$app.ModelName
    #move application to correct folders
    $rt=Move-CMApp -appDefObject $appDefObject -globalDefObject $globalDefObject 
    #deploy application to collection
    $rt=start-cmDeployment -appDefObject $appDefObject -globalDefObject $globalDefObject 
    #machine policy an testgruppe
    Invoke-CMClientNotification -DeviceCollectionName $globalDefObject.includeCollection -NotificationType RequestMachinePolicyNow
  }
  else
  {
    write-host "GlobalDef doConfigMgrIntegration set to false. No ConfigMgr integration."
  }
}


 export-modulemember *-*

