[CmdletBinding()]
Param (
  [Parameter(Mandatory=$true)]
  [string]$appDefinition,
  [switch]$ignoreHashError
)
Get-Module | Where-Object {$_.ModuleType -eq 'Script'} | Remove-Module
import-module .\automateAppDeployment.psm1 

function Get-GlobaleDefinition {
  $globalDef=New-Module -AsCustomObject -ScriptBlock {
    [string]$site=$null
    [string]$siteServer=$null
    [string]$appDeploySource=$null
    [string]$targetBasePath=$null
    [string]$targetUNCBasePath=$null
    [string]$distributionPointGroupName=$null
    [string]$includeCollection=$null
    [string]$rootFolder=$null
    [int]$updateCollectionMin=0
    [string]$limitingCollection=$null
    [string]$namespace=$null

    [bool]$doConfigMgrIntegration=$true
    Export-ModuleMember -Variable *
  }
  $globalDef.appDeploySource='D:\_Scripting\PSAppDeployToolkit\Toolkit'
  $globalDef.targetBasePath='D:\ConfigMgr\Content\Software' 
  $globalDef.targetUNCBasePath='\\cm01\ConfigMgr\Content\Software'
  $globalDef.rootFolder='_Software'
  $globalDef.DistributionPointGroupName='Intranet'
  $globalDef.includeCollection='Test Clients' 
  $globalDef.site='MBK'
  $globalDef.siteServer='cm01'
  $globalDef.updateCollectionMin=30
  $globalDef.limitingCollection='All Systems'
  $globalDef.namespace='root\SMS\Site_'+$globalDef.site
  $globalDef.doConfigMgrIntegration=$true
  $globalDef
}

function getAppName{
  [CmdletBinding()]
  param(
    [Parameter(Mandatory=$false)]
    [string]$vendor,
    [Parameter(Mandatory=$true)]
    [string]$software,
    [Parameter(Mandatory=$true)]
    [string]$version,
    [Parameter(Mandatory=$false)]
    [bool]$isX86
    
  )
  
  #sonderfall: wenn der Softwarename bereits die versionnummer beinhaltet, dann nicht nochmal hinzufuegen
  if ($software.Contains($version)) {
    $colName=$vendor + ' '+$software
  }
  else
  {
    $colName=$vendor+' '+$software + ' ' + $version
  }
  if ($isX86 -ne $null) {
    if ($isX86) {
      $colname+=' x86'
    }
    else
    {
      $colname+=' x64'
    }
  }
  $colName.trim() 
}

function Get-AppDefinition{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$false)]
    [xml]$xml,
    [Parameter(Mandatory=$false)]
    [string]$sourcefiles,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  $appDef = New-Module -AsCustomObject -ScriptBlock {
    [string]$vendor=$null
    [string]$software=$null
    [string]$version=$null
    [bool]$isX86=$false
    #path to source files (same as appdefinition.xml folder)
    [string]$sourcefiles=$null
    [string]$targetFiles=$null
    [string]$uncTargetFiles=$null
    #full path to main file
    [string]$mainFile=$null
    #only filename of main file
    [string]$mainFileShort=$null
    [string]$install=$null
    [string]$uninstall=$null
    [string]$detect=$null
    [string]$appShortName=$null
    [string]$appLongName=$null
    [string]$msiProductCode=$null
    [string]$applicationObjectID=$null

    #name of hash algorithmen - currently only sha256
    [string]$hashType=$null
    #result of hash of main file to compare
    [string]$hashValue=$null

    #Format: Key: SHA256:234jk Value: uninstall=$true/$false
    [hashtable]$supersedence=@{}
    #Format: Key: Group Name, Value: [hashtable] Key: SHA256:234234, Value: autoinstall=$true/$False
    [hashtable]$dependency=@{}

    [bool]$hasUninstall=$true
    Export-ModuleMember -Variable *
  }
  
    #transfer values from xml file to object
  $appDef.sourcefiles=$sourcefiles
  
  $appDef.isX86=($xml.appdefinition.info.isX86 -ieq 'true')
  $appDef.hasUninstall=-not ($xml.appdefinition.info.hasUninstall -ieq 'false')
  
  $appDef.mainFile=Join-Path $appDef.sourceFiles ($xml.appdefinition.file)
  $appDef.mainFileShort=$xml.appdefinition.file
  if ($xml.appdefinition.info.setupType -eq 'MSI') {
    #extract msi table for infos...
    $msiTable=Get-MsiTableProperty -Path $appDef.mainFile -Table 'Property' -ContinueOnError $false
    
    $appDef.vendor=$msiTable.Manufacturer 
    $appDef.software=$msiTable.ProductName
    $appDef.version=$msiTable.ProductVersion
    $appDef.msiProductCode=$msiTable.ProductCode
    #region msp auswerten
  } elseif ($xml.appdefinition.info.setupType -eq 'MSP') {
    #source: http://www.software-virtualisierung.de/entry/msppowershell.html
        $wi = New-Object -com WindowsInstaller.Installer
        $mspdb = $wi.GetType().InvokeMember("OpenDatabase", "InvokeMethod", $Null, $wi, $($appDef.mainFile, 32))
        $su = $mspdb.GetType().InvokeMember("SummaryInformation", "GetProperty", $Null, $mspdb, $Null)
        #proerty 7: https://msdn.microsoft.com/en-us/library/windows/desktop/aa372043%28v=vs.85%29.aspx
        $appDef.msiProductCode=$su.GetType().InvokeMember("Property", "GetProperty", $Null, $su, 7)
        #patchcode:
        #$su.GetType().InvokeMember("Property", "GetProperty", $Null, $su, 9)
        #{AC76BA86-7AD7-FFFF-2550-AE0F0675A700}{AC76BA86-7AD7-FFFF-2550-AE0F06756C00}{AC76BA86-7AD7-FFFF-2550-AE0F06758E00}{AC76BA86-7AD7-FFFF-2550-AE0F06759000}{AC76BA86-7AD7-FFFF-2550-AE0F06759100}
  }
  #endregion
  else { #exe auslesen
    $vers=(get-item $appDef.mainFile).VersionInfo
    if ($vers -ne $null) {
      if ($vers.CompanyName -ne $null) {$appDef.vendor=$vers.CompanyName.trim()}
      if ($vers.ProductName -ne $null) {$appDef.software=$vers.ProductName.trim()}
      if ($vers.ProductVersion -ne $null) {$appDef.version=$vers.ProductVersion.trim()}
      if ($appDef.version -eq $null) {
        $appDef.version=$vers.FileVersion.trim()
      }
    }
  }
  


  #Company aus msi berschreiben, wenn es explizit im xml definiert ist
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'company' -and -not $_.IsEmpty} | ForEach-Object {$appDef.vendor=$_.InnerText}
  #software aus msi berschreiben, wenn es explizit im xml definiert ist
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'productName' -and -not $_.IsEmpty} | ForEach-Object {$appDef.software=$_.InnerText}
  #version aus msi berschreiben, wenn es explizit im xml definiert ist
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'productVersion' -and -not $_.IsEmpty} | ForEach-Object {$appDef.version=$_.InnerText}
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'msiProductCode' -and -not $_.IsEmpty} | ForEach-Object {$appDef.msiProductCode=$_.InnerText}
  
  
  $xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'install' -and -not $_.IsEmpty} | ForEach-Object { $appDef.install=[Environment]::NewLine+($_.InnerText)}
  $xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'uninstall' -and -not $_.IsEmpty} | ForEach-Object { $appDef.uninstall=[Environment]::NewLine+($_.InnerText)}
  $xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'detection' -and -not $_.IsEmpty} | ForEach-Object { $appDef.detect=[Environment]::NewLine+($_.InnerText)}
  
  #read hash values
  $xml.appdefinition.ChildNodes | where {$_.name -eq "hash" -and -not $_.isEmpty} | ForEach-Object {
    $appdef.hashValue=$_.InnerText
    $appdef.hashType=$_.Type
  }

  $appDef.appShortName=getAppName -vendor '' -software $appdef.Software -version $appdef.Version -isX86 $appdef.isX86 
  $appDef.appLongName=getAppName -vendor $appDef.Vendor -software $appdef.Software -version $appdef.Version -isX86 $appdef.isX86 
  $appDef.uncTargetFiles=[IO.Path]::getFullPath((Join-path(Join-Path $globalDefObject.targetUNCBasePath $appDef.vendor) ($appDef.appShortName)))
  $appDef.targetFiles=[IO.Path]::getFullPath((Join-path(Join-Path $globalDefObject.targetBasePath $appDef.vendor) $appDef.appShortName))
  $xml.appdefinition.supersedence.deploymentType | ForEach-Object {
    $appdef.supersedence.add(($_.type+":"+$_.InnerText),($_.uninstall -ieq 'true'))
  }

  $xml.appdefinition.dependency.dependencyGroup | ForEach-Object {
    $name=$_.name
    if ($appDef.dependency[$name] -eq $null) {
        [hashtable]$values=@{}
        $appDef.dependency.add($name,$values)
    }
    else
    {
        [hashtable]$values=$appDef.dependency.get[$name]
    }
    $_.application | ForEach-Object {
        $values.add(($_.type+":"+$_.InnerText),($_.autoInstall -ieq 'true'))
    }
  }

  $appDef
}

function create-application{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  ) 
    #create filesystem folders and appdeployment toolkit
  new-ApplicationSource  -appDefObject $appDefObject -globalDefObject $globalDefObject
  if ($globalDefObject.doConfigMgrIntegration) {
    #create new folder and collection in configmgr
    $collection=new-SoftwareCollection -appDefObject $appDefObject -globalDefObject $globalDefObject
    #create new application in configmgr
    new-CMApplicationDetect -appDefObject $appDefObject -globalDefObject $globalDefObject
    $app=Get-CMApplication -Name $appDefObject.appLongName
    $appDefObject.applicationObjectID=$app.ModelName
    #move application to correct folders
    Move-CMApp -appDefObject $appDefObject -globalDefObject $globalDefObject 
    #deploy application to collection
    start-cmDeployment -appDefObject $appDefObject -globalDefObject $globalDefObject 
  }
  else
  {
    write-host "GlobalDef doConfigMgrIntegration set to false. No ConfigMgr integration."
  }
}

function New-MSPSetup{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  ) 
  
  

  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    if (-not $appDefObject.isX86) {
      $appDefObject.detect='Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {$_.psChildname -like "'+$appDefObject.msiProductCode+'" -and $_.DisplayVersion -eq "'+$appDefObject.Version+'" -and $_.WindowsInstaller -eq 1}'
    }
    else
    {
      $appDefObject.detect='if ([IntPtr]::Size -eq 8) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
      $appDefObject.detect+=[Environment]::NewLine+' Get-ItemProperty $regPath | where {$_.psChildname -like "'+$appDefObject.msiProductCode+'" -and $_.DisplayVersion -eq "'+$appDefObject.Version+'" -and $_.WindowsInstaller -eq 1}'
    }
  }

  create-application -appDefObject $appDefObject -globalDefObject $globalDefObject

}


function New-MSISetup{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  ) 
  
  

  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    if (-not $appDefObject.isX86) {
      $appDefObject.detect='Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {$_.psChildname -like "'+$appDefObject.msiProductCode+'" -and $_.DisplayVersion -eq "'+$appDefObject.Version+'" -and $_.WindowsInstaller -eq 1}'
    }
    else
    {
      $appDefObject.detect='if ([IntPtr]::Size -eq 8) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
      $appDefObject.detect+=[Environment]::NewLine+' Get-ItemProperty $regPath | where {$_.psChildname -like "'+$appDefObject.msiProductCode+'" -and $_.DisplayVersion -eq "'+$appDefObject.Version+'" -and $_.WindowsInstaller -eq 1}'
    }
  }

  create-application -appDefObject $appDefObject -globalDefObject $globalDefObject

}

function New-InnoSetup {
  param
  (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  
  $mainFile=$appDefObject.mainFile
  

  if ($appDefObject.install -eq $null -or $appDefObject.install -eq '')
  {
    write-host 'No InnoSetup install parameter found. Generating script...'
    $appDefObject.install=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Installing '+($appDefObject.software)+' '+($appDefObject.version)+'. This may take some time. Please wait..."'
    $appDefObject.install+=[Environment]::NewLine+'$nsisLog=join-path $logDirectory ($installName + "_" + $appDeployToolkitName + "_" + $deploymentType + "_NSIS.log")'
    $appDefObject.install+=[Environment]::NewLine+'Execute-Process -Path "'+$appDefObject.mainFileShort+'" -Parameters "/VERYSILENT /NORESTART /LOG=`"$nsisLog`"" -WindowStyle Hidden'
  }
  
  
  if ($appDefObject.uninstall -eq $null -or $appDefObject.uninstall -eq '')
  {
    write-host 'No InnoSetup UnInstall parameter found. Generating script...'
    $appDefObject.uninstall=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Uninstalling '+($appDefObject.software)+' '+($appDefObject.version)+'. This may take some time. Please wait..."'
    $appDefObject.uninstall+=[Environment]::NewLine+'$uninstallApp=(Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {$_.DisplayName -like "*'+$appDefObject.software+'*" -and $_.DisplayVersion -eq "'+$appDefObject.version+'"}).UninstallString.replace("`"","")'
    $appDefObject.uninstall+=[Environment]::NewLine+'$nsisLog=join-path $logDirectory ($installName + "_" + $appDeployToolkitName + "_" + $deploymentType + "_NSIS.log")'
    $appDefObject.uninstall+=[Environment]::NewLine+'Execute-Process -Path "$uninstallApp" -Parameters "/VERYSILENT /NORESTART /LOG=$nsisLog" -WindowStyle Hidden'
  }
  
  
  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    write-host 'No InnoSetup detect parameter found. Generating script...'
    $appDefObject.detect="Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {`$_.DisplayName -like `"*"+$appDefObject.software+"*`" -and `$_.DisplayVersion -eq `""+$appDefObject.version+"`"}"
  }


  create-application -appDefObject $appDefObject -globalDefObject $globalDefObject
}

function New-NSISSetup {
  param
  (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  
  $mainFile=$appDefObject.mainFile
  


  if ($appDefObject.install -eq $null -or $appDefObject.install -eq '')
  {
    write-host 'No NSIS install parameter found. Generating script...'
    $appDefObject.install=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Installing '+($appDefObject.software)+' '+($appDefObject.version)+'. This may take some time. Please wait..."'
    $appDefObject.install+=[Environment]::NewLine+'Execute-Process -Path "'+$appDefObject.mainFileShort+'" -Parameters "/S" -WindowStyle Hidden'
  }
  
  if ($appDefObject.uninstall -eq $null -or $appDefObject.uninstall -eq '')
  {
    $appDefObject.uninstall=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Uninstalling '+($appDefObject.software)+' '+($appDefObject.version)+'. This may take some time. Please wait..."'
    if ($appDefObject.isX86) {
      $appDefObject.uninstall+=[Environment]::NewLine+'if ($is64Bit) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
    }
    else
    {
      $appDefObject.uninstall+=[Environment]::NewLine+'$regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"'
    }
    $appDefObject.uninstall+=[Environment]::NewLine+'$uninstallApp=(Get-ItemProperty $regPath | where {$_.DisplayName -like "*'+$appDefObject.software+'*" -and $_.DisplayVersion -eq "'+$appDefObject.version+'"}).UninstallString.replace("`"","")'
    $appDefObject.uninstall+=[Environment]::NewLine+'Execute-Process -Path "$uninstallApp" -Parameters "/S" -WindowStyle Hidden'
  }
  
  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    if ($appDefObject.isX86) {
      $appDefObject.detect='if ([IntPtr]::Size -eq 8) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
    }
    else
    {
      $appDefObject.detect='$regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"'
    }
    $appDefObject.detect+=[Environment]::NewLine+'Get-ItemProperty $regPath | where {$_.DisplayName -like "*'+$appDefObject.software+'*" -and $_.DisplayVersion -eq "'+$appDefObject.version+'"}'
  }
   
  create-application -appDefObject $appDefObject -globalDefObject $globalDefObject 
}

function check-mainfile{
  param
  (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  $realHash=Get-FileHash -Path $appDefObject.mainFile -Algorithm $appdefObject.hashType
  $realHash.hash -ieq $appDefObject.hashValue
}

[xml]$xml=Get-Content $appDefinition

$sourceFiles=(get-item $appdefinition).DirectoryName
$globalDef=get-GlobaleDefinition

$appDef=get-AppDefinition -xml $xml -sourcefiles $sourceFiles -globalDefObject $globalDef

write-host 'Globale Definition:'
$globalDef
write-host 'Application Definition:'
$appDef 

if ((check-mainfile -appDefObject $appDef -globalDefObject $globalDef) -or $ignoreHashError) {
    if (-not $ignoreHashError) {
        write-host -BackgroundColor Green -ForegroundColor Black "Hash value is correct. Creating application."
    }
    if ($xml.appdefinition.info.setupType -eq 'InnoSetup') {
        new-InnoSetup -appDefObject $appDef -globalDefObject $globalDef
    }
    elseif ($xml.appdefinition.info.setupType -eq 'NSIS') {
        new-NSISSetup -appDefObject $appDef -globalDefObject $globalDef
    }
    elseif ($xml.appdefinition.info.setupType -eq 'MSI') {
        new-MSISetup -appDefObject $appDef -globalDefObject $globalDef
    }
    elseif ($xml.appdefinition.info.setupType -eq 'MSP') {
        new-MSPSetup -appDefObject $appDef -globalDefObject $globalDef
    }
    write-host 'Resulting Application Definition:'
    $appDef
}
else
{
  write-host -BackgroundColor Red ("Error: Hash value of main file " + $appdef.mainfile + " doesn't match with value in XML file "+$appDefinition+" (you can ignore this error with the switch -ignoreHashError)")
}


