[CmdletBinding()]
Param (
  [Parameter(Mandatory=$true)]
  [string]$appDefinition
)
Get-Module | Where-Object {$_.ModuleType -eq 'Script'} | Remove-Module
import-module .\automateAppDeployment.psm1 

function Get-GlobaleDefinition {
  $globalDef=New-Module -AsCustomObject -ScriptBlock {
    [string]$site=$null
    [string]$siteServer=$null
    [string]$appDeploySource=$null
    [string]$targetBasePath=$null
    [string]$targetUNCBasePath=$null
    [string]$distributionPointGroupName=$null
    [string]$includeCollection=$null
    [string]$rootFolder=$null
    [int]$updateCollectionMin=0
    [string]$limitingCollection=$null
    [string]$namespace=$null
    Export-ModuleMember -Variable *
  }
  $globalDef.appDeploySource='D:\_Scripting\PSAppDeployToolkit\Toolkit'
  $globalDef.targetBasePath='D:\ConfigMgr\Content\Software' 
  $globalDef.targetUNCBasePath='\\cm01\ConfigMgr\Content\Software'
  $globalDef.rootFolder='_Software'
  $globalDef.DistributionPointGroupName='Intranet'
  $globalDef.includeCollection='Test Clients' 
  $globalDef.site='MBK'
  $globalDef.siteServer='cm01'
  $globalDef.updateCollectionMin=30
  $globalDef.limitingCollection='All Systems'
  $globalDef.namespace='root\SMS\Site_'+$globalDef.site
  $globalDef
}

function getAppName{
  [CmdletBinding()]
  param(
    [Parameter(Mandatory=$false)]
    [string]$vendor,
    [Parameter(Mandatory=$true)]
    [string]$software,
    [Parameter(Mandatory=$true)]
    [string]$version,
    [Parameter(Mandatory=$false)]
    [bool]$isX86
    
  )
  
  #sonderfall: wenn der Softwarename bereits die versionnummer beinhaltet, dann nicht nochmal hinzufuegen
  if ($software.Contains($version)) {
    $colName=$vendor + ' '+$software
  }
  else
  {
    $colName=$vendor+' '+$software + ' ' + $version
  }
  if ($isX86 -ne $null) {
    if ($isX86) {
      $colname+=' x86'
    }
    else
    {
      $colname+=' x64'
    }
  }
  $colName.trim() 
}

function Get-AppDefinition{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$false)]
    [xml]$xml,
    [Parameter(Mandatory=$false)]
    [string]$sourcefiles,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  $appDef = New-Module -AsCustomObject -ScriptBlock {
    [string]$vendor=$null
    [string]$software=$null
    [string]$version=$null
    [bool]$isX86=$false
    [string]$sourcefiles=$null
    [string]$targetFiles=$null
    [string]$uncTargetFiles=$null
    [string]$mainFile=$null
    [string]$mainFileShort=$null
    [string]$install=$null
    [string]$uninstall=$null
    [string]$detect=$null
    [string]$appShortName=$null
    [string]$appLongName=$null
    [string]$msiProductCode=$null
    [string]$applicationObjectID=$null
    [bool]$hasUninstall=$true
    Export-ModuleMember -Variable *
  }
  
  $appDef.sourcefiles=$sourcefiles
  
  $appDef.isX86=($xml.appdefinition.info.isX86 -ieq 'true')
  $appDef.hasUninstall=-not ($xml.appdefinition.info.hasUninstall -ieq 'false')
  
  $appDef.mainFile=Join-Path $appDef.sourceFiles ($xml.appdefinition.file)
  $appDef.mainFileShort=$xml.appdefinition.file
  if ($xml.appdefinition.info.setupType -eq 'MSI') {
    #extract msi table for infos...
    $msiTable=Get-MsiTableProperty -Path $appDef.mainFile -Table 'Property' -ContinueOnError $false
    
    $appDef.vendor=$msiTable.Manufacturer 
    $appDef.software=$msiTable.ProductName
    $appDef.version=$msiTable.ProductVersion
    $appDef.msiProductCode=$msiTable.ProductCode
  }
  else {
    $vers=(get-item $appDef.mainFile).VersionInfo
    if ($vers -ne $null) {
      if ($vers.CompanyName -ne $null) {$appDef.vendor=$vers.CompanyName.trim()}
      if ($vers.ProductName -ne $null) {$appDef.software=$vers.ProductName.trim()}
      if ($vers.ProductVersion -ne $null) {$appDef.version=$vers.ProductVersion.trim()}
      if ($appDef.version -eq $null) {
        $appDef.version=$vers.FileVersion.trim()
      }
    }
  }
  
  #Company aus msi berschreiben, wenn es explizit im xml definiert ist
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'company' -and -not $_.IsEmpty} | ForEach-Object {$appDef.vendor=$_.InnerText}
  #software aus msi berschreiben, wenn es explizit im xml definiert ist
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'productName' -and -not $_.IsEmpty} | ForEach-Object {$appDef.software=$_.InnerText}
  #version aus msi berschreiben, wenn es explizit im xml definiert ist
  $xml.appdefinition.info.ChildNodes | Where-Object {$_.name -eq 'productVersion' -and -not $_.IsEmpty} | ForEach-Object {$appDef.version=$_.InnerText}
  
  
  $xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'install' -and -not $_.IsEmpty} | ForEach-Object { $appDef.install=[Environment]::NewLine+($_.InnerText)}
  $xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'uninstall' -and -not $_.IsEmpty} | ForEach-Object { $appDef.uninstall=[Environment]::NewLine+($_.InnerText)}
  $xml.appdefinition.ChildNodes | Where-Object {$_.name -eq 'detection' -and -not $_.IsEmpty} | ForEach-Object { $appDef.detect=[Environment]::NewLine+($_.InnerText)}
  
  $appDef.appShortName=getAppName -vendor '' -software $appdef.Software -version $appdef.Version -isX86 $appdef.isX86 
  $appDef.appLongName=getAppName -vendor $appDef.Vendor -software $appdef.Software -version $appdef.Version -isX86 $appdef.isX86 
  $appDef.uncTargetFiles=[IO.Path]::getFullPath((Join-path(Join-Path $globalDefObject.targetUNCBasePath $appDef.vendor) ($appDef.appShortName)))
  $appDef.targetFiles=[IO.Path]::getFullPath((Join-path(Join-Path $globalDefObject.targetBasePath $appDef.vendor) $appDef.appShortName))
  $appDef
}

function New-MSISetup{
  [CmdletBinding()]
  Param (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  ) 
  
  $collection=new-SoftwareCollection -appDefObject $appDefObject -globalDefObject $globalDefObject
  

  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    if (-not $appDefObject.isX86) {
      $appDefObject.detect='Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {$_.psChildname -like "'+$appDefObject.msiProductCode+'" -and $_.DisplayVersion -eq "'+$appDefObject.Version+'" -and $_.WindowsInstaller -eq 1}'
    }
    else
    {
      $appDefObject.detect='if ([IntPtr]::Size -eq 8) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
      $appDefObject.detect+=[Environment]::NewLine+' Get-ItemProperty $regPath | where {$_.psChildname -like "'+$appDefObject.msiProductCode+'" -and $_.DisplayVersion -eq "'+$appDefObject.Version+'" -and $_.WindowsInstaller -eq 1}'
    }
  }

  new-ApplicationSource  -appDefObject $appDefObject -globalDefObject $globalDefObject
  
  $app=new-CMApplicationDetect -appDefObject $appDefObject -globalDefObject $globalDefObject
  
  $appDefObject.applicationObjectID=$app.ModelName
  Move-CMApp -appDefObject $appDefObject -globalDefObject $globalDefObject 
  
  start-cmDeployment -appDefObject $appDefObject -globalDefObject $globalDefObject 
}

function New-InnoSetup {
  param
  (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  
  $mainFile=$appDefObject.mainFile
  
  $collection=new-SoftwareCollection -appDefObject $appDefObject -globalDefObject $globalDefObject
  if ($appDefObject.install -eq $null -or $appDefObject.install -eq '')
  {
    write-host 'No InnoSetup install parameter found. Generating script...'
    $appDefObject.install=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Installing '+($appDefObject.software)+' '+($appDefObject.version)+'. This may take some time. Please wait..."'
    $appDefObject.install+=[Environment]::NewLine+'$nsisLog=join-path $logDirectory ($installName + "_" + $appDeployToolkitName + "_" + $deploymentType + "_NSIS.log")'
    $appDefObject.install+=[Environment]::NewLine+'Execute-Process -Path "'+$appDefObject.mainFileShort+'" -Parameters "/VERYSILENT /NORESTART /LOG="$nsisLog" -WindowStyle Hidden'
  }
  
  
  if ($appDefObject.uninstall -eq $null -or $appDefObject.uninstall -eq '')
  {
    write-host 'No InnoSetup UnInstall parameter found. Generating script...'
    $appDefObject.uninstall=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Uninstalling '+($appDefObject.software)+' '+($appDefObject.version)+'. This may take some time. Please wait..."'
    $appDefObject.uninstall+=[Environment]::NewLine+'$uninstallApp=(Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {$_.DisplayName -like "*'+$appDefObject.software+'*" -and $_.DisplayVersion -eq "'+$appDefObject.version+'"}).UninstallString.replace("`"","")'
    $appDefObject.uninstall+=[Environment]::NewLine+'$nsisLog=join-path $logDirectory ($installName + "_" + $appDeployToolkitName + "_" + $deploymentType + "_NSIS.log")'
    $appDefObject.uninstall+=[Environment]::NewLine+'Execute-Process -Path "$uninstallApp" -Parameters "/VERYSILENT /NORESTART /LOG=$nsisLog" -WindowStyle Hidden'
  }
  
  
  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    write-host 'No InnoSetup detect parameter found. Generating script...'
    $appDefObject.detect="Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* | where {`$_.DisplayName -like `"*"+$appDefObject.software+"*`" -and `$_.DisplayVersion -eq `""+$appDefObject.version+"`"}"
  }

  new-ApplicationSource -appDefObject $appDefObject -globalDefObject $globalDefObject
  
  $app=new-CMApplicationDetect -appDefObject $appDefObject -globalDefObject $globalDefObject
  
  $appDefObject.applicationObjectID=$app.ModelName
  Move-CMApp -appDefObject $appDefObject -globalDefObject $globalDefObject 
  
  start-cmDeployment -appDefObject $appDefObject -globalDefObject $globalDefObject 
  
}

function New-NSISSetup {
  param
  (
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$appDefObject,
    [Parameter(Mandatory=$true)]
    [PSCustomObject]$globalDefObject
  )
  
  $mainFile=$appDefObject.mainFile
  
  $collection=new-SoftwareCollection -appDefObject $appDefObject -globalDefObject $globalDefObject

  if ($appDefObject.install -eq $null -or $appDefObject.install -eq '')
  {
    write-host 'No NSIS install parameter found. Generating script...'
    $appDefObject.install=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Installing '+($appDefObject.software)+' '+($appDefObject.version)+'. This may take some time. Please wait..."'
    $appDefObject.install+=[Environment]::NewLine+'Execute-Process -Path "'+$appDefObject.mainFileShort+'" -Parameters "/S" -WindowStyle Hidden'
  }
  
  if ($appDefObject.uninstall -eq $null -or $appDefObject.uninstall -eq '')
  {
    $appDefObject.uninstall=[Environment]::NewLine+'Show-InstallationProgress -StatusMessage "Uninstalling '+($appDefObject.software)+' '+($appDefObject.version)+'. This may take some time. Please wait..."'
    if ($appDefObject.isX86) {
      $appDefObject.uninstall+=[Environment]::NewLine+'if ($is64Bit) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
    }
    else
    {
      $appDefObject.uninstall+=[Environment]::NewLine+'$regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"'
    }
    $appDefObject.uninstall+=[Environment]::NewLine+'$uninstallApp=(Get-ItemProperty $regPath | where {$_.DisplayName -like "*'+$appDefObject.software+'*" -and $_.DisplayVersion -eq "'+$appDefObject.version+'"}).UninstallString.replace("`"","")'
    $appDefObject.uninstall+=[Environment]::NewLine+'Execute-Process -Path "$uninstallApp" -Parameters "/S" -WindowStyle Hidden'
  }
  
  if ($appDefObject.detect -eq $null -or $appDefObject.detect -eq '')
  {
    if ($appDefObject.isX86) {
      $appDefObject.detect='if ([IntPtr]::Size -eq 8) {$regPath="HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"} else { $regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"}'
    }
    else
    {
      $appDefObject.detect='$regPath="HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"'
    }
    $appDefObject.detect+=[Environment]::NewLine+'Get-ItemProperty $regPath | where {$_.DisplayName -like "*'+$appDefObject.software+'*" -and $_.DisplayVersion -eq "'+$appDefObject.version+'"}'
  }
   
  new-ApplicationSource -appDefObject $appDefObject -globalDefObject $globalDefObject
  
  $app=new-CMApplicationDetect -appDefObject $appDefObject -globalDefObject $globalDefObject
  
  $appDefObject.applicationObjectID=$app.ModelName
  Move-CMApp -appDefObject $appDefObject -globalDefObject $globalDefObject 
  
  start-cmDeployment -appDefObject $appDefObject -globalDefObject $globalDefObject 
  
}


[xml]$xml=Get-Content $appDefinition

$sourceFiles=(get-item $appdefinition).DirectoryName
$globalDef=get-GlobaleDefinition

$appDef=get-AppDefinition -xml $xml -sourcefiles $sourceFiles -globalDefObject $globalDef

write-host 'Globale Definition:'
$globalDef
write-host 'Application Definition:'
$appDef 

if ($xml.appdefinition.info.setupType -eq 'InnoSetup') {
  new-InnoSetup -appDefObject $appDef -globalDefObject $globalDef
}
elseif ($xml.appdefinition.info.setupType -eq 'NSIS') {
  new-NSISSetup -appDefObject $appDef -globalDefObject $globalDef
}
elseif ($xml.appdefinition.info.setupType -eq 'MSI') {
  new-MSISetup -appDefObject $appDef -globalDefObject $globalDef
}
write-host 'Resulting Application Definition:'
$appDef


