﻿using System.Diagnostics;
using System.ServiceProcess;
using System.Threading;
using System.IO;
using System.Text;
using System;

namespace NagiosService
{
    /// <summary>
    /// Copyright (C) 2009 Markus Bäker
    /// This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; 
    /// either version 3 of the License, or (at your option) any later version.
    /// This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
    /// without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
    /// You should have received a copy of the GNU General Public License along with this program; if not, see <http://www.gnu.org/licenses/>.
    /// 
    /// Diese Klasse stellt den Windows Dienst dar.
    /// Hierin werden alle Timer und Umgebungsvariablen
    /// gestartet.
    /// </summary>
    partial class NagiosService : ServiceBase
    {
        // A variable for the timer object that is used to periodically check for new alerts.
        private Timer chkAlertsTimer = null;
        private Timer chkHeartbeatTimer = null;
        private Timer chkResendTimer = null;
        /// <summary>
        /// Alle 10 Sekunden nach neuen Alarmen suchen.
        /// </summary>
        private int pollingIntervalAlerts = 10000; //10 sec
        /// <summary>
        /// Im Eventlog wird alle 5 Minuten ein Eintrag erzeugt, 
        /// der zeigt, dass der Dienst noch läuft (kann in einem
        /// Management Pack überprüft werden=
        /// </summary>
        private int heartBeatTimer = 300000;
        /// <summary>
        /// Nicht sofort beim Start alle Verbindungen/Alarme aufbauen.
        /// </summary>
        private int startupDelayMS = 30000;
        /// <summary>
        /// Alle 5 Minuten werden alle offenen Alarme erneut an Nagios
        /// geschickt.
        /// </summary>
        private int resendAlertsMS = 300000;
        private int alertCounter = 0;

        // Read Tracelevel from app.config and remember the value
        TraceSwitch _DebugLevel = new TraceSwitch("DebugLevel", "The Output level of tracing");

        private SCOMAdapter scom = new SCOMAdapter();

        public NagiosService()
        {
            InitializeComponent();
        }

        /// <summary>
        /// Einsprungspunkt für Timer, um Alarme erneut
        /// an Nagios zu versenden.
        /// </summary>
        /// <param name="data"></param>
        private void resendQueue(object data)
        {
            scom.resendQueue();
            EventLog.WriteEntry("Nagios-SCOMConnector resent alerts: " + scom.getResendQueueSize(), EventLogEntryType.Information, 990);
        }

        /// <summary>
        /// Einsprungspunkt für Timer um den Heartbeat ins
        /// Eventlog zu schreiben
        /// </summary>
        /// <param name="data"></param>
        private void heartBeat(object data)
        {
            EventLog.WriteEntry("Nagios-SCOMConnector is still alive. Overall alerts received: " + alertCounter + ", Resend queue: " + scom.getResendQueueSize(), EventLogEntryType.Information, 997);
            scom.reopenConnector();
        }

        /// <summary>
        /// Einsprungspunkt für Timer um die Alerts aus dem 
        /// OpsMgr zu lesen und an Nagios zu verschicken.
        /// </summary>
        /// <param name="data"></param>
        private void checkEvents(object data)
        {
            if (scom.openConnection())
            {
                scom.openConnector();
                EventLog.WriteEntry("OpsMgr Connection opened.", EventLogEntryType.Information);
            }

            int alerts = scom.checkEvents();
            if (alerts > 0)
            {
                alertCounter += alerts;
                //EventLog.WriteEntry("Nagios Connector retrieved  " + alerts + " from OpsMgr", EventLogEntryType.Information);

            }
        }

        /// <summary>
        /// Methode, die das Logging aktiviert.
        /// Logdateien werden im All Users Profile unter Anwendungsdaten abgelegt.
        /// Der Loglevel wird in der config Datei definiert.
        /// </summary>
        private void initalizeLogging()
        {
            //SOURCE: http://www.outlookcode.com/codedetail.aspx?id=1405
            // define the datastore for application specific files

            //LOCALAPPDATA
            string applicationPath =
                Path.Combine(System.Environment.GetEnvironmentVariable("ALLUSERSPROFILE"),
                "Application Data\\kit\\nagiosConnector");

            // Create Directory if it doesn't exists
            Directory.CreateDirectory(applicationPath);
            Directory.CreateDirectory(Path.Combine(applicationPath, "logs"));

            // define the logging destination
            string logFile = Path.Combine(applicationPath, "logs\\tracelog.txt");
            // Configure TraceListener
            Trace.AutoFlush = true;
            Trace.Listeners.Add(new TextWriterTraceListener(logFile, "NagiosTraceListener"));


            Trace.WriteLineIf(_DebugLevel.TraceInfo, String.Format("{0}: Application (Version: {1}) startup.", DateTime.Now, System.Reflection.Assembly.GetExecutingAssembly().GetName().Version));
        }

        /// <summary>
        /// Einsprungsmethode für den Dienst
        /// </summary>
        /// <param name="args"></param>
        protected override void OnStart(string[] args)
        {
            initalizeLogging();
            EventLog.WriteEntry("Starting Nagios Connector with parameters: " + NagiosAdapter.getConfig(), EventLogEntryType.Information);

            string pathToExe = Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().GetModules()[0].FullyQualifiedName);
            Directory.SetCurrentDirectory(pathToExe);

            Trace.WriteLineIf(_DebugLevel.TraceInfo, String.Format("{0}: Setting the working directory {1} to the path to the exe {2}.", DateTime.Now,Directory.GetCurrentDirectory(),pathToExe));


            // Create the delegate that invokes methods for the timer.
            TimerCallback runChkAlertsDelegate = new TimerCallback(checkEvents);

            // Create a timer that runs at the configured polling interval. Delay start of timer based on configuration value.   
            chkAlertsTimer = new Timer(runChkAlertsDelegate, null, startupDelayMS, pollingIntervalAlerts);

            // Create the delegate that invokes methods for the timer.
            TimerCallback runHeartbeatDelegate = new TimerCallback(heartBeat);

            // Create a timer that runs at the configured polling interval. Delay start of timer based on configuration value.   
            chkHeartbeatTimer = new Timer(runHeartbeatDelegate, null, 5, heartBeatTimer);

            // Create the delegate that invokes methods for the timer.
            TimerCallback runResendDelegate = new TimerCallback(resendQueue);

            // Create a timer that runs at the configured polling interval. Delay start of timer based on configuration value.   
            chkResendTimer = new Timer(runResendDelegate, null, 10000, resendAlertsMS);

        }


        protected override void OnStop()
        {
            // TODO: Add code here to perform any tear-down necessary to stop your service.
        }
    }
}
