﻿using System;
using System.Collections.Generic;
using System.Text;
using Microsoft.EnterpriseManagement;
using Microsoft.EnterpriseManagement.Common;
using Microsoft.EnterpriseManagement.ConnectorFramework;
using System.Collections.ObjectModel;
using System.Collections;
using System.Threading;
using System.Configuration;
using System.Diagnostics;

namespace NagiosService
{
    /// <summary>
    /// Copyright (C) 2009 Markus Bäker
    /// This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; 
    /// either version 3 of the License, or (at your option) any later version.
    /// This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
    /// without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
    /// You should have received a copy of the GNU General Public License along with this program; if not, see <http://www.gnu.org/licenses/>.
    /// 
    /// </summary>
    class SCOMAdapter
    {
        public static Guid connectorGuid = new Guid("{038F2550-B6F3-11DD-BBCC-4EF555D89593}"); // Created online... (random)
        private MonitoringConnector connector=null;
        private ManagementGroup mg=null;
        private ConnectorFrameworkAdministration cfAdmin = null;
        private System.Collections.Hashtable itemsSend = new Hashtable();
        TraceSwitch _DebugLevel = new TraceSwitch("DebugLevel", "The Output level of tracing");

        public int getResendQueueSize()
        {
            return itemsSend.Count;
        }

        /// <summary>
        /// Die noch offenen Alarme werden regelmäßig an den Nagios gesandt, 
        /// um ein zurücksetzen des Fehlerstatus zu verhindern.
        /// Beispiel: Zwei Alarme zu einem Rechner werden generiert. Der SCOM 
        /// Dienst des Rechners im Nagios ist somit kritisch. Wird ein Alarm geschlossen,
        /// so wird dieser an Nagios geschickt und der Rechner wechselt zurück auf Grün, obwohl
        /// der andere Alarm noch offen ist.
        /// </summary>
        public void resendQueue()
        {
            NagiosAdapter nagios = new NagiosAdapter();

            //kopie der einträge machen, da parallel die liste verändert werden könnte (threads)
            ArrayList values = new ArrayList(itemsSend.Values);
            foreach (string line in values)
            {
                nagios.sendToNagios(line);
            }
        }

        /// <summary>
        /// Verbindung zum SDK neu aufbauen
        /// Wird regelmäßig gemacht, um eventuelle Unterbrechungen wieder
        /// zu korrigieren
        /// </summary>
        public void reopenConnector()
        {
            connector = null;
            mg = null;
            openConnection();
            openConnector();
        }

        /// <summary>
        /// Öffnet die Verbindung zum lokalen OpsMgr Server
        /// (Somit muss der Connector auf dem OpsMgr installiert werden.)
        /// Dadurch sind keine extra Credentials notwendig, da das lokale System unter
        /// der der Dienst läuft die passenden Berechtigungen hat.
        /// </summary>
        /// <returns>true/false ob Verbindungsaufbau notwendig war (eventuell verbindung schon offen)</returns>
        public bool openConnection()
        {
            if (mg == null || !mg.IsConnected)
            {
                mg = new ManagementGroup("localhost");
                Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: Connection to ManagementGroup on localhost opened.", DateTime.Now));
                cfAdmin = mg.GetConnectorFrameworkAdministration();
                return true;
            }
            return false;
        }

        /// <summary>
        /// Verbindet sich über die bereits offene Connection zum Management Server
        /// zu dem eigentlichen Connector unter angabe der eigenen Connector GUID.
        /// </summary>
        public void openConnector()
        {
            connector = cfAdmin.GetMonitoringConnector(connectorGuid);
        }

        /// <summary>
        /// Installiert den Connector im OpsMgr.
        /// Er nennt sich selber "Nagios Connector"
        /// </summary>
        public void installConnector()
        {

                        ConnectorInfo info = new ConnectorInfo();

                        info.Description = "Nagios Connector";
                        info.DisplayName = "Nagios Connector";
                        info.Name = "Nagios Connector";

                        connector = cfAdmin.Setup(info, connectorGuid);

                        connector.Initialize();

                        Console.WriteLine("Created {0} with ID: {1}", connector.Name, connector.Id);
        }

        /// <summary>
        /// Versucht den Connector wieder aus dem OpsMgr zu entfernen und dabei alle
        /// eintragenen Subscriptions zu löschen.
        /// </summary>
        public void uninstallConnector()
        {
                        connector = cfAdmin.GetMonitoringConnector(connectorGuid);
                        ReadOnlyCollection<MonitoringConnectorSubscription> subscriptions;

                        subscriptions = cfAdmin.GetConnectorSubscriptions();

                        foreach (MonitoringConnectorSubscription subscription in subscriptions)
                        {
                            if (subscription.MonitoringConnectorId == connectorGuid)
                            {
                                cfAdmin.DeleteConnectorSubscription(subscription);
                            }
                        }

                        connector.Uninitialize();
                        cfAdmin.Cleanup(connector);

                        Console.WriteLine("Connector removed.");
                    }

        /// <summary>
        /// Zentrale Methode:
        /// Überprüft ob im OpsMgr in der Warteschlange zum Nagios Connector neue
        /// Events aufgetaucht sind.
        /// Diese werden dann heruntergeladen und verarbeitet.
        /// Dabei werden die Daten umformatiert und in ein Nagios-Kompatibles Format
        /// umgewandelt. Am Ende wird der Alert über den NagiosAdapter an Nagios geschickt und
        /// in die interne Resend Queue gespeichert.
        /// </summary>
        /// <returns>Anzahl der neuen Alerts</returns>
        public int checkEvents()
        {
            NagiosAdapter nagios = new NagiosAdapter();
            ReadOnlyCollection<ConnectorMonitoringAlert> alerts;
            int i = 1;

            if (connector != null && connector.Initialized)
            {
                try
                {
                    alerts = connector.GetMonitoringAlerts();

                    if (alerts.Count > 0)
                    {
                        //muss bestätigt werden, da sonst auch bereits geschlossene alerts wieder auftauchen
                        connector.AcknowledgeMonitoringAlerts(alerts);
                    }


                    foreach (ConnectorMonitoringAlert alert in alerts)
                    {


                        // Here you can send the alert to the other management system.

                        //sendnsca.vbs State: $Data/Context/DataItem/ResolutionStateName$  Alert: $Data/Context/DataItem/AlertName$ Source: $Data/Context/DataItem/ManagedEntityDisplayName$ Severity: $Data/Context/DataItem/Severity$ Path: $Data/Context/DataItem/ManagedEntityPath$\
                        //Create nagios string

                        Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: Alert Received: Path: {1}, Name: {2}, Severity: {3}, ResolutionState: {4}", DateTime.Now, alert.MonitoringObjectPath, alert.Name, alert.Severity, alert.ResolutionState));

                        NagiosAlert nAlert = new Transformer().transform(alert);

                        string line = nagios.getNagiosLine(nAlert);

                        if (alert.ResolutionState.ToString().Equals("255"))
                        {
                            //bei closed aler aus der resend queue herausnehmen
                            if (itemsSend.Contains(alert.Id))
                            {
                                itemsSend.Remove(alert.Id);
                            }
                        }
                        else
                        {
                            //alle neuen events (nicht closed) in hashmap zum resend.
                            if (!itemsSend.Contains(alert.Id))
                            {
                                itemsSend.Add(alert.Id, line);
                            }
                            else
                            {
                                //hähh??? Alert zweimal geöffnet?
                            }
                        }

                        Console.WriteLine(line);
                        nagios.sendToNagios(line);
                        i = i + 1;
                    }
                }
                catch (Exception e)
                {
                    Trace.WriteLineIf(_DebugLevel.TraceWarning, String.Format("{0}: Error while getting new alerts from scom: {1}.", DateTime.Now,e.Message));
                    connector.Reconnect(mg);
                }
            }
            else
            {
                Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: connection to scom is null or not initialized.", DateTime.Now));
            }
            return i - 1; //startet with 1
        }
    }
}
