﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using Microsoft.EnterpriseManagement;
using Microsoft.EnterpriseManagement.Common;
using Microsoft.EnterpriseManagement.ConnectorFramework;
using System.Diagnostics;
namespace NagiosService
{
    class Processor
    {
        TraceSwitch _DebugLevel = new TraceSwitch("DebugLevel", "The Output level of tracing");
        private XmlDocument getDefinition()
        {
            if (Properties.Settings.Default.processDefinition == null || Properties.Settings.Default.processDefinition.Trim().Length == 0)
            {
                Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: Processor: No processing definition set.", DateTime.Now));
                return null;
            }
            XmlDocument doc = new XmlDocument();

            // Load the XML data from a file.
            // This code assumes that the XML file is in the same folder.
            doc.Load(Properties.Settings.Default.processDefinition);
            Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: Processor: Processing definition {1} loaded.", DateTime.Now, Properties.Settings.Default.processDefinition));

            return doc;
        }

        private string getSCOMField(string field, ConnectorMonitoringAlert inputAlert)
        {
            string value = null;
            switch (field)
            {
                case "path": value = inputAlert.MonitoringObjectPath; break;
                case "netbios": value = inputAlert.NetbiosComputerName; break;
                case "fullname": value = inputAlert.MonitoringObjectFullName; break;
                case "severity": value = inputAlert.Severity.ToString(); break;
                case "state": value = inputAlert.ResolutionState.ToString(); break;
                case "name": value = inputAlert.Name; break;
                case "description": value = inputAlert.Description; break;
                case "alertid": value = inputAlert.Id.ToString(); break;
                default: return null; //no matching -> wrong input definition
            }
            return value;

        }

        private bool checkInput(XmlNode input, ConnectorMonitoringAlert inputAlert)
        {
            string value = "";
            Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: Processor: checkInput: field: {1}, compare: {2}, value: {3}", DateTime.Now,
                input.Attributes.GetNamedItem("field").Value, input.Attributes.GetNamedItem("compare").Value, input.InnerText));
            value = getSCOMField(input.Attributes.GetNamedItem("field").Value, inputAlert);

            if (value == null)
            {
                Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: Processor: value is null. cannot compare.", DateTime.Now));
                return false;
            }
            string compare = input.InnerText;
            switch (input.Attributes.GetNamedItem("compare").Value)
            {
                case "contains": return value.Contains(compare);
                case "containsIgnoreCase": return value.ToLower().Contains(compare.ToLower());
                case "equals": return value.Equals(input.Value);
                case "equalsIgnoreCase": return value.ToLower().Equals(input.Value.ToLower());
                default: return false;
            }
        }

        private void manipulateOutput(XmlNode output, ref NagiosAlert outputAlert)
        {
            string value = output.InnerText;

            Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: Processor: manipulateOutput: {1}", DateTime.Now, output.OuterXml));

            if (output.Attributes.GetNamedItem("case") != null)
            {
                Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: Processor: manipulateOutput: changing case: {1}", DateTime.Now, output.Attributes.GetNamedItem("case").Value));
                switch (output.Attributes.GetNamedItem("case").Value.ToLower())
                {
                    case "lower": value = value.ToLower(); break;
                    case "upper": value = value.ToUpper(); break;
                    default: Trace.WriteLineIf(_DebugLevel.TraceWarning, String.Format("{0}: Processor: manipulateOutput: unknown case: {1}", DateTime.Now, output.Attributes.GetNamedItem("case").Value)); break;
                }
            }
            switch (output.Attributes.GetNamedItem("field").Value)
            {
                case "text": outputAlert.Text = value; break;
                case "host": outputAlert.Host = value; break;
                case "service": outputAlert.Service = value; break;
                case "state": outputAlert.State = int.Parse(value); break; //can create an execption!
                default: Trace.WriteLineIf(_DebugLevel.TraceWarning, String.Format("{0}: Processor: manipulateOutput: unknown field: {1}", DateTime.Now, output.Attributes.GetNamedItem("field").Value)); break;
            }
        }

        public void process(ConnectorMonitoringAlert inputAlert, ref NagiosAlert outputAlert)
        {
            XmlDocument doc = getDefinition();
            if (doc == null)
            {
                //no instructions defined -> change nothing
                return;
            }

            foreach (XmlNode node in doc.GetElementsByTagName("instruction"))
            {
                XmlNode input = null;
                XmlNode output = null;
                foreach (XmlNode childNode in node.ChildNodes)
                {
                    //really ugly xml processing...
                    if (childNode.Name.Equals("input"))
                    {
                        input = childNode;
                    }
                    if (childNode.Name.Equals("output"))
                    {
                        output = childNode;
                    }

                }
                if (input != null && output != null && checkInput(input, inputAlert))
                {
                    if (output.InnerText.StartsWith("$"))
                    {
                        string value = getSCOMField(output.InnerText.Substring(1), inputAlert);
                        if (value != null)
                        {
                            Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: Processor: process replaces target value with {1}.", DateTime.Now, value));

                            output.InnerText = value;
                        }
                        else
                        {
                            Trace.WriteLineIf(_DebugLevel.TraceVerbose, String.Format("{0}: Processor: process found variable but couldn't replace it: {1}.", DateTime.Now, output.InnerText));

                        }

                    }

                    manipulateOutput(output, ref outputAlert);
                }
            }
        }


    }
}
