﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Configuration;

namespace NagiosService
{
    /// <summary>
    /// Copyright (C) 2009 Markus Bäker
    /// This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; 
    /// either version 3 of the License, or (at your option) any later version.
    /// This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
    /// without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
    /// You should have received a copy of the GNU General Public License along with this program; if not, see <http://www.gnu.org/licenses/>.
    /// 
    /// Zentraler Adapter zu Nagios.
    /// Er wertet die Einstellungen und führt die Hinterlegte Kommandozeile aus, um
    /// die Alerts an Nagios weiterzureichen.
    /// </summary>
    class NagiosAdapter
    {
        // Read Tracelevel from app.config and remember the value
        TraceSwitch _DebugLevel = new TraceSwitch("DebugLevel", "The Output level of tracing");

        public string getNagiosLine(NagiosAlert alert)
        {
            return getNagiosLine(alert.Host, alert.Service, alert.State, alert.Text);
        }

        public string getNagiosLine(string host, int state, string text)
        {
            return getNagiosLine(host, Properties.Settings.Default.NagiosSCOMService, state, text);
        }

        /// <summary>
        /// Kombiniert die Parameter zu einer korrekten send_ncsa Alarmzeile
        /// </summary>
        /// <param name="host">Name der alarmerzeugenden Hosts</param>
        /// <param name="service">Welcher Nagios-Dienst hat den Alarm erzeugt?</param>
        /// <param name="state">Status des Alarms</param>
        /// <param name="text">beschreibender Text</param>
        /// <returns>String</returns>
        public string getNagiosLine(string host, string service, int state, string text)
        {
            string rt;
            rt = host + ";" + service + ";" + state + ";" + text+"\n";
            return rt;
        }

        /// <summary>
        /// Gibt die aktuelle Nagioskonfiguration für Debugzwecke aus
        /// </summary>
        /// <returns>String</returns>
        public static string getConfig()
        {
            string rt = String.Format("Current nagios config:\nHost: {0}\nPort: {1}\nncsa file:{2}\nConfig file:{3}\n", Properties.Settings.Default.NagiosHost, Properties.Settings.Default.NagiosPort, Properties.Settings.Default.ncsaPath, Properties.Settings.Default.ncsaConfigFile);
            return rt;
        }

        /// <summary>
        /// Sendet den Alarm an Nagios. 
        /// Alle notwendigen Parameter werden aus der Anwendungskonfiguration
        /// ausgelesen.
        /// </summary>
        /// <param name="line">Der zu sendende Alarm (Rückgabe von getNagiosLine)</param>
        public void sendToNagios(string line)
        {
            sendToNagios(Properties.Settings.Default.NagiosHost, Properties.Settings.Default.NagiosPort,line);
            if (Properties.Settings.Default.NagiosHost1 != null && Properties.Settings.Default.NagiosHost1.Trim().Length>0)
            {
                Trace.WriteLineIf(_DebugLevel.TraceVerbose,String.Format("{1}: Second Nagios: >>{0}<<",Properties.Settings.Default.NagiosHost1,DateTime.Now));
                sendToNagios(Properties.Settings.Default.NagiosHost1, Properties.Settings.Default.NagiosPort1, line);
            }
        }

        /// <summary>
        /// Eigentliche Methode, die den Alarm versendet. Dabei können
        /// unterschiedliche Zielhosts angegeben werden.
        /// </summary>
        /// <param name="server"></param>
        /// <param name="port"></param>
        /// <param name="line"></param>
        public void sendToNagios(string server, int port, string line)
        {
            string args = "-H " + server + " -d ; -c \""+Properties.Settings.Default.ncsaConfigFile+"\" -p " + port;
            Trace.WriteLineIf(_DebugLevel.TraceInfo, String.Format("{0}: Send to Nagios {3} commandline:  {1}, Message: {2}", DateTime.Now, args, line, Properties.Settings.Default.ncsaPath));

            ProcessStartInfo psi = new ProcessStartInfo(Properties.Settings.Default.ncsaPath, args);
            psi.RedirectStandardOutput = false;
            psi.RedirectStandardInput = true;
            psi.UseShellExecute = false;
            psi.WindowStyle = ProcessWindowStyle.Hidden;

            Process ncsa = Process.Start(psi);
            System.IO.StreamWriter input = ncsa.StandardInput;
            input.WriteLine(line);
            input.Close();
            ncsa.WaitForExit(10000);
            if (ncsa.HasExited)
            {
                Trace.WriteLineIf(_DebugLevel.TraceInfo, String.Format("{0}: Event send to nagios.",DateTime.Now));
            }

        }
    }
}
